<?php

namespace App\Http\Controllers;

use App\Jobs\Vrm\SendMail;
use App\Mail\SendEmail;
use App\Models\User;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\Mail;

class TestEmailController extends Controller
{
    /** Set your email here for testing (used when config mail.test_recipient is not set). */
    protected string $testRecipient = '';

    /**
     * Build test email details used by both direct send and job.
     *
     * @return array<string, mixed>
     */
    protected function testEmailDetails(): array
    {
        return [
            'logo' => null,
            'title' => 'Email test',
            'subtitle' => 'Diagnostic test from FxZidii',
            'body' => '<p>If you receive this, the delivery path used for this test is working.</p>',
            'outro' => '<p>Direct send = SMTP/transport. Job send = queue/cron.</p>',
            'btn' => ['title' => null, 'link' => null],
            'btn_extra' => null,
        ];
    }

    /**
     * Send test email directly (no queue). Use to verify SMTP/server.
     */
    public function sendDirect(): Response
    {
        $recipient = config('mail.test_recipient') ?? $this->testRecipient;

        if (! $recipient) {
            return response('Set $testRecipient in TestEmailController.', 400);
        }

        try {
            $mailable = new SendEmail(
                _details: $this->testEmailDetails(),
                _subject: '[FxZidii] Test email (direct send – SMTP)'
            );
            Mail::to($recipient)->send($mailable);

            return response("Test email sent directly to {$recipient}. Check your inbox. (This tests SMTP/transport.)");
        } catch (\Throwable $e) {
            return response("Direct send failed: {$e->getMessage()}", 500);
        }
    }

    /**
     * Send test email via SendMail job. Use to verify queue/cron.
     */
    public function sendViaJob(): Response
    {
        $recipient = config('mail.test_recipient') ?? $this->testRecipient;

        if (! $recipient) {
            return response('Set $testRecipient in TestEmailController.', 400);
        }

        try {
            $mailable = new SendEmail(
                _details: $this->testEmailDetails(),
                _subject: '[FxZidii] Test email (via job – queue/cron)'
            );
            SendMail::dispatch($recipient, mailable: $mailable);

            return response("Test email queued for {$recipient}. Run the queue worker (or wait for cron), then check your inbox. (This tests job/queue/cron.)");
        } catch (\Throwable $e) {
            return response("Job dispatch failed: {$e->getMessage()}", 500);
        }
    }

    /**
     * Test the same flow as registration: trigger SendVerifyEmailJob via User::sendEmailVerificationNotification().
     * Finds a user by $testRecipient email and queues the verification email (and SMS) as on signup.
     */
    public function sendVerificationEmail(): Response
    {
        $recipient = config('mail.test_recipient') ?? $this->testRecipient;

        if (! $recipient) {
            return response('Set $testRecipient in TestEmailController.', 400);
        }

        $user = User::query()->where('email', $recipient)->first();

        if (! $user) {
            return response("No user found with email {$recipient}. Use an existing user's email in \$testRecipient.", 404);
        }

        try {
            $user->sendEmailVerificationNotification();

            return response("Verification email (and SMS) queued for {$recipient} (user id: {$user->id}). Same flow as registration. Run the queue worker, then check inbox.");
        } catch (\Throwable $e) {
            return response("Failed to queue verification: {$e->getMessage()}", 500);
        }
    }
}
