<?php

namespace App\Mail;

use App\Services\ReceiptPdfService;
use Illuminate\Bus\Queueable;
use Illuminate\Mail\Mailable;
use Illuminate\Mail\Mailables\Attachment;
use Illuminate\Mail\Mailables\Content;
use Illuminate\Mail\Mailables\Envelope;
use Illuminate\Queue\SerializesModels;

class WithdrawalRequestStatusMail extends Mailable
{
    use Queueable;
    use SerializesModels;

    public function __construct(
        public string $recipientName,
        public string $currencyCode,
        public float $amount,
        public string $status,
        public ?string $reference = null,
        public ?string $phoneOrBankAccountNumber = null,
        public ?string $occurredAt = null,
    ) {}

    /**
     * Get the message envelope.
     */
    public function envelope(): Envelope
    {
        $statusLabel = ucfirst($this->status);

        return new Envelope(
            subject: "Withdrawal Request {$statusLabel}",
        );
    }

    /**
     * Get the message content definition.
     */
    public function content(): Content
    {
        $statusLabel = ucfirst($this->status);

        return new Content(
            view: 'emails.simple-message',
            with: [
                'greeting' => 'Hello '.$this->recipientName.',',
                'intro' => "Your withdrawal request has been {$statusLabel}.",
                'lines' => [
                    'Amount: '.$this->currencyCode.' '.number_format($this->amount, 2),
                    'Reference: '.($this->reference ?: 'N/A'),
                ],
                'outro' => 'If you need support, please contact us.',
            ],
        );
    }

    /**
     * Get the attachments for the message.
     *
     * @return array<int, \Illuminate\Mail\Mailables\Attachment>
     */
    public function attachments(): array
    {
        if (! $this->shouldAttachReceipt()) {
            return [];
        }

        $timestamp = now()->format('YmdHis');

        return [
            Attachment::fromData(fn (): string => app(ReceiptPdfService::class)->generateWalletTransactionReceiptPdf(
                fullName: $this->recipientName,
                type: 'withdrawal',
                currencyCode: $this->currencyCode,
                amount: $this->amount,
                reference: $this->reference,
                phoneOrBankAccountNumber: $this->phoneOrBankAccountNumber,
                occurredAt: $this->occurredAt,
            ), "receipt-withdrawal-{$timestamp}.pdf")->withMime('application/pdf'),
        ];
    }

    protected function shouldAttachReceipt(): bool
    {
        return in_array(strtolower($this->status), ['approved', 'completed', 'successful', 'success'], true);
    }
}
