<?php

namespace App\Notifications;

use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class AdminDepositNotification extends Notification implements ShouldQueue
{
    use Queueable;

    public function __construct(
        protected string $memberName,
        protected string $memberEmail,
        protected float $amount,
        protected string $currencyCode,
        protected string $status,
        protected string $channel = 'wallet',
        protected ?string $reference = null,
    ) {}

    /**
     * Get the notification's delivery channels.
     *
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        return ['mail'];
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail(object $notifiable): MailMessage
    {
        $statusLabel = ucfirst($this->status);
        $channelLabel = strtoupper($this->channel);

        return (new MailMessage)
            ->subject("Deposit {$statusLabel}: {$this->currencyCode} ".number_format($this->amount, 2))
            ->greeting('Hello Admin,')
            ->line("A member deposit has been {$this->status}.")
            ->line("Member: {$this->memberName} ({$this->memberEmail})")
            ->line("Channel: {$channelLabel}")
            ->line("Amount: {$this->currencyCode} ".number_format($this->amount, 2))
            ->line('Status: '.$statusLabel)
            ->line('Reference: '.($this->reference ?: 'N/A'))
            ->action('Open Wallet Requests', route('admin.wallet-requests.index'))
            ->line('Please review and monitor this transaction.');
    }

    /**
     * Get the array representation of the notification.
     *
     * @return array<string, mixed>
     */
    public function toArray(object $notifiable): array
    {
        return [
            'member_name' => $this->memberName,
            'member_email' => $this->memberEmail,
            'amount' => $this->amount,
            'currency_code' => $this->currencyCode,
            'status' => $this->status,
            'channel' => $this->channel,
            'reference' => $this->reference,
        ];
    }
}
