<?php

namespace App\Notifications;

use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class WalletTransactionNotification extends Notification implements ShouldQueue
{
    use Queueable;

    public function __construct(
        protected string $type,
        protected string $currencyCode,
        protected float $amount,
        protected string $status = 'completed',
    ) {}

    /**
     * Get the notification's delivery channels.
     *
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        return ['mail'];
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail(object $notifiable): MailMessage
    {
        $typeLabel = ucfirst(str_replace('_', ' ', $this->type));
        $statusLabel = ucfirst($this->status);

        return (new MailMessage)
            ->subject("Wallet {$typeLabel}: {$statusLabel}")
            ->greeting("Hello {$notifiable->name},")
            ->line("A wallet transaction was recorded on your account: {$typeLabel}.")
            ->line("Amount: {$this->currencyCode} ".number_format($this->amount, 2))
            ->line("Status: {$statusLabel}")
            ->action('Open Account', route('member.account'))
            ->line('If you did not initiate this activity, contact support immediately.');
    }

    /**
     * Get the array representation of the notification.
     *
     * @return array<string, mixed>
     */
    public function toArray(object $notifiable): array
    {
        return [
            'type' => $this->type,
            'currency_code' => $this->currencyCode,
            'amount' => $this->amount,
            'status' => $this->status,
        ];
    }
}
