<?php

namespace App\Services;

use App\Models\Security;

class DatabaseNSE implements NSEService
{
    public function __construct(
        private readonly YahooNSEOutputHelper $outputHelper
    ) {}

    /**
     * @return array<int|string, mixed>
     */
    public function getStocks(): array
    {
        $rows = Security::query()
            ->where('is_active', true)
            ->with('latestLog')
            ->orderBy('trading_name')
            ->get()
            ->map(fn (Security $security): array => $this->toRow($security))
            ->all();

        return $this->outputHelper->formatRows($rows, 'Stocks list retrieved.');
    }

    /**
     * @return array<int|string, mixed>
     */
    public function getStock(string $symbol): array
    {
        $normalizedSymbol = strtoupper(trim($symbol));
        if ($normalizedSymbol === '') {
            return [];
        }

        $security = Security::query()
            ->where('trading_name', $normalizedSymbol)
            ->where('is_active', true)
            ->with('latestLog')
            ->first();

        if (! $security instanceof Security) {
            return $this->outputHelper->formatRows([[
                'ticker' => $normalizedSymbol,
                'name' => $normalizedSymbol,
                'volume' => '0',
                'price' => '0.00',
                'change' => '+0.00',
            ]], 'Stock data retrieved.');
        }

        return $this->outputHelper->formatRows([$this->toRow($security)], 'Stock data retrieved.');
    }

    /**
     * @return array{ticker: string, name: string, volume: string, price: string, change: string}
     */
    private function toRow(Security $security): array
    {
        return [
            'ticker' => strtoupper((string) $security->trading_name),
            'name' => (string) ($security->name ?: $security->trading_name),
            'volume' => '0',
            'price' => number_format($security->latestPrice(), 2, '.', ''),
            'change' => $this->formatSignedChange($security->latestChangePercent()),
        ];
    }

    private function formatSignedChange(float $value): string
    {
        $prefix = $value >= 0 ? '+' : '';

        return $prefix.number_format($value, 2, '.', '');
    }
}
