<?php

namespace App\Services\Trading;

use App\Jobs\Vrm\SendMail;
use App\Mail\AdminTransactionMail;
use App\Mail\SecurityOrderStatusMail;
use App\Models\PortfolioHolding;
use App\Models\SecurityOrder;
use App\Models\WalletTransaction;
use App\Services\Sms\SmsNotifier;
use Illuminate\Support\Facades\DB;

class OrderApprovalService
{
    public function approve(SecurityOrder $order): void
    {
        $order->loadMissing(['user', 'security', 'wallet']);

        if ($order->status !== 'pending') {
            throw new \RuntimeException(__('Order not available for approval.'));
        }

        DB::transaction(function () use ($order): void {
            $wallet = $order->wallet;
            $orderType = strtolower((string) $order->order_type);

            if (! $wallet) {
                throw new \RuntimeException('Wallet not found for this order.');
            }

            if ($orderType === 'sell') {
                $holding = PortfolioHolding::query()->firstOrNew([
                    'user_id' => $order->user_id,
                    'security_id' => $order->security_id,
                ]);
                $existingQuantity = (int) ($holding->quantity ?? 0);
                $reservedQuantity = (int) ($holding->reserved_quantity ?? 0);
                $sellQuantity = (int) $order->quantity;

                if ($existingQuantity < $sellQuantity) {
                    throw new \RuntimeException('Insufficient holding quantity for sell order.');
                }

                if ($reservedQuantity < $sellQuantity) {
                    throw new \RuntimeException('Reserved holding quantity is insufficient for this sell order.');
                }

                $averageBuyPrice = (float) ($holding->average_buy_price ?? 0);
                $costBasis = $averageBuyPrice * $sellQuantity;
                $profitLossAmount = (float) $order->total_amount - $costBasis;
                $profitLossPercent = $costBasis > 0 ? ($profitLossAmount / $costBasis) * 100 : 0;

                $remainingQuantity = $existingQuantity - $sellQuantity;
                $remainingInvested = max(0, (float) ($holding->total_invested ?? 0) - $costBasis);
                $remainingReserved = max(0, $reservedQuantity - $sellQuantity);

                if ($remainingQuantity === 0 && $remainingReserved === 0) {
                    $holding->delete();
                } else {
                    $holding->quantity = $remainingQuantity;
                    $holding->reserved_quantity = $remainingReserved;
                    $holding->total_invested = $remainingInvested;
                    $holding->average_buy_price = $remainingQuantity > 0 ? $remainingInvested / $remainingQuantity : 0;
                    $holding->save();
                }

                $wallet->balance = (float) $wallet->balance + (float) $order->total_amount;
                $wallet->save();

                $order->profit_loss_amount = $profitLossAmount;
                $order->profit_loss_percent = $profitLossPercent;
                $order->is_active = false;

                WalletTransaction::query()->create([
                    'wallet_id' => $wallet->id,
                    'user_id' => $order->user_id,
                    'type' => 'order_credit',
                    'status' => 'completed',
                    'amount' => $order->total_amount,
                    'currency_code' => $wallet->currency_code,
                    'description' => "Sell order approved for {$order->security->trading_name}",
                    'meta' => [
                        'order_id' => $order->id,
                        'profit_loss_amount' => $profitLossAmount,
                        'profit_loss_percent' => $profitLossPercent,
                    ],
                    'approved_by' => auth()->id(),
                    'approved_at' => now(),
                ]);
            } else {
                $wallet->used_amount = max(0, (float) $wallet->used_amount - (float) $order->total_amount);
                $wallet->balance = max(0, (float) $wallet->balance - (float) $order->total_amount);
                $wallet->save();

                $holding = PortfolioHolding::query()->firstOrNew([
                    'user_id' => $order->user_id,
                    'security_id' => $order->security_id,
                ]);

                $existingQuantity = (int) ($holding->quantity ?? 0);
                $newQuantity = $existingQuantity + (int) $order->quantity;
                $existingInvested = (float) ($holding->total_invested ?? 0);
                $newInvested = $existingInvested + (float) $order->total_amount;

                $holding->quantity = $newQuantity;
                $holding->total_invested = $newInvested;
                $holding->average_buy_price = $newQuantity > 0 ? $newInvested / $newQuantity : 0;
                $holding->save();

                $order->profit_loss_amount = 0;
                $order->profit_loss_percent = 0;
                $order->is_active = true;

                WalletTransaction::query()->create([
                    'wallet_id' => $wallet->id,
                    'user_id' => $order->user_id,
                    'type' => 'order_debit',
                    'status' => 'completed',
                    'amount' => $order->total_amount,
                    'currency_code' => $wallet->currency_code,
                    'description' => "Order approved for {$order->security->trading_name}",
                    'meta' => ['order_id' => $order->id],
                    'approved_by' => auth()->id(),
                    'approved_at' => now(),
                ]);
            }

            $order->status = 'approved';
            $order->approved_by = auth()->id();
            $order->approved_at = now();
            $order->traded_at = $order->traded_at ?? now();
            $order->volume = max(1, (int) $order->volume ?: (int) $order->quantity);
            $order->save();
        });

        SendMail::dispatch($order->user->email, mailable: new SecurityOrderStatusMail(
            $order->user->name,
            $order->security->name,
            'approved',
            (int) $order->quantity,
            $order->currency_code,
            (float) $order->total_amount,
        ), notificationType: 'user_transaction');

        $adminEmail = config('mail.admin_address');
        if (filled($adminEmail)) {
            SendMail::dispatch((string) $adminEmail, mailable: new AdminTransactionMail(
                $order->user->name,
                $order->user->email,
                (float) $order->total_amount,
                $order->currency_code,
                strtolower((string) $order->order_type) === 'buy' ? 'shares_buy' : 'shares_sell',
                'approved',
                'trade',
                (string) $order->id,
            ), notificationType: 'admin_transaction');
        }

        app(SmsNotifier::class)->sendToPhone(
            $order->user->phone,
            "Order approved for {$order->security->name}. Qty: {$order->quantity}.",
            'user_transaction'
        );
    }

    public function reject(SecurityOrder $order): void
    {
        $order->loadMissing(['user', 'security', 'wallet']);

        if ($order->status !== 'pending') {
            throw new \RuntimeException(__('Order not available for rejection.'));
        }

        DB::transaction(function () use ($order): void {
            $wallet = $order->wallet;
            $orderType = strtolower((string) $order->order_type);

            if (! $wallet) {
                throw new \RuntimeException('Wallet not found for this order.');
            }

            if ($orderType === 'buy') {
                $wallet->used_amount = max(0, (float) $wallet->used_amount - (float) $order->total_amount);
                $wallet->save();
            }

            if ($orderType === 'sell') {
                $holding = PortfolioHolding::query()->where([
                    'user_id' => $order->user_id,
                    'security_id' => $order->security_id,
                ])->first();

                if ($holding) {
                    $holding->reserved_quantity = max(0, (int) ($holding->reserved_quantity ?? 0) - (int) $order->quantity);
                    $holding->save();
                }
            }

            $order->status = 'rejected';
            $order->approved_by = auth()->id();
            $order->approved_at = now();
            $order->is_active = false;
            $order->save();

            if ($orderType === 'buy') {
                WalletTransaction::query()->create([
                    'wallet_id' => $wallet->id,
                    'user_id' => $order->user_id,
                    'type' => 'order_credit',
                    'status' => 'completed',
                    'amount' => $order->total_amount,
                    'currency_code' => $wallet->currency_code,
                    'description' => "Order rejected for {$order->security->trading_name}; funds released",
                    'meta' => ['order_id' => $order->id],
                    'approved_by' => auth()->id(),
                    'approved_at' => now(),
                ]);
            }
        });

        SendMail::dispatch($order->user->email, mailable: new SecurityOrderStatusMail(
            $order->user->name,
            $order->security->name,
            'rejected',
            (int) $order->quantity,
            $order->currency_code,
            (float) $order->total_amount,
        ), notificationType: 'user_transaction');

        $adminEmail = config('mail.admin_address');
        if (filled($adminEmail)) {
            SendMail::dispatch((string) $adminEmail, mailable: new AdminTransactionMail(
                $order->user->name,
                $order->user->email,
                (float) $order->total_amount,
                $order->currency_code,
                strtolower((string) $order->order_type) === 'buy' ? 'shares_buy' : 'shares_sell',
                'rejected',
                'trade',
                (string) $order->id,
            ), notificationType: 'admin_transaction');
        }

        app(SmsNotifier::class)->sendToPhone(
            $order->user->phone,
            "Order rejected for {$order->security->name}. Qty: {$order->quantity}.",
            'user_transaction'
        );
    }
}
