<?php

namespace App\Services\Vrm;

use App\Jobs\Vrm\SendMail;

class InvestmentEmailService
{
    public function sendSubscriptionCreated(
        string $email,
        string $recipientName,
        string $packageName,
        string $currencyCode,
        float $amount,
    ): void {
        SendMail::dispatch(
            $email,
            $this->subscriptionCreatedDetails($recipientName, $packageName, $currencyCode, $amount),
            __('Investment subscription created'),
            notificationType: 'user_transaction',
        );
    }

    public function sendInvestmentClaimed(
        string $email,
        string $recipientName,
        string $currencyCode,
        float $amount,
    ): void {
        SendMail::dispatch(
            $email,
            $this->investmentClaimedDetails($recipientName, $currencyCode, $amount),
            __('Investment return claimed'),
            notificationType: 'user_transaction',
        );
    }

    /**
     * @return array<string, mixed>
     */
    protected function subscriptionCreatedDetails(
        string $recipientName,
        string $packageName,
        string $currencyCode,
        float $amount,
    ): array {
        return [
            'logo' => null,
            'title' => __('Investment Subscription Created'),
            'subtitle' => __('Hello :name,', ['name' => $recipientName]),
            'body' => implode('<br>', [
                __('Your investment subscription has been activated.'),
                __('Package: :name', ['name' => $packageName]),
                __('Amount: :currency :amount', ['currency' => $currencyCode, 'amount' => number_format($amount, 2)]),
            ]),
            'outro' => __('Thank you for using our application!'),
            'btn' => [
                'title' => __('View investments'),
                'link' => $this->memberInvestmentsUrl(),
            ],
            'btn_extra' => null,
        ];
    }

    /**
     * @return array<string, mixed>
     */
    protected function investmentClaimedDetails(
        string $recipientName,
        string $currencyCode,
        float $amount,
    ): array {
        return [
            'logo' => null,
            'title' => __('Investment Return Claimed'),
            'subtitle' => __('Hello :name,', ['name' => $recipientName]),
            'body' => implode('<br>', [
                __('You have successfully claimed your investment return.'),
                __('Amount credited: :currency :amount', ['currency' => $currencyCode, 'amount' => number_format($amount, 2)]),
            ]),
            'outro' => __('Thank you for using our application!'),
            'btn' => [
                'title' => __('View account'),
                'link' => $this->memberAccountUrl(),
            ],
            'btn_extra' => null,
        ];
    }

    protected function memberInvestmentsUrl(): string
    {
        return app()->bound('url')
            ? route('member.investments')
            : rtrim((string) config('app.url'), '/').'/member/investments';
    }

    protected function memberAccountUrl(): string
    {
        return app()->bound('url')
            ? route('member.account')
            : rtrim((string) config('app.url'), '/').'/member/account';
    }
}
