<?php

use App\Models\InvestmentPackage;
use App\Traits\Vrm\Livewire\WithNotifications;
use Livewire\Attributes\Computed;
use Livewire\Component;
use Livewire\WithPagination;

new class extends Component {
    use WithNotifications;
    use WithPagination;

    public string $search = '';

    public int $perPage = 10;

    public function updatedSearch(): void
    {
        $this->resetPage();
    }

    public function updatedPerPage(): void
    {
        $this->resetPage();
    }

    #[Computed]
    public function results()
    {
        $query = InvestmentPackage::query();

        if ($this->search !== '') {
            $query->where('name', 'like', '%'.$this->search.'%');
        }

        return $query->orderByDesc('created_at')->paginate($this->perPage, ['*'], 'packages_page');
    }

    public function activate(int $id): void
    {
        $package = InvestmentPackage::query()->find($id);
        if (! $package) {
            $this->notifyError(__('Package not found.'));

            return;
        }
        $package->is_active = true;
        $package->save();
        $this->notifySuccess(__('Package was activated successfully.'));
    }

    public function deactivate(int $id): void
    {
        $package = InvestmentPackage::query()->find($id);
        if (! $package) {
            $this->notifyError(__('Package not found.'));

            return;
        }
        $package->is_active = false;
        $package->save();
        $this->notifySuccess(__('Package was deactivated successfully.'));
    }
}; ?>

<div>
    <x-admin-panel>
        <x-slot name="header">{{ __('Investment Packages') }}</x-slot>
        <x-slot name="desc">
            {{ __('Manage investment packages. Members invest in packages; each package has min/max amount, interest and duration.') }}
        </x-slot>
        <x-slot name="button">
            <a href="{{ route('admin.investment-packages.create') }}" wire:navigate
                class="bg-blue-500 dark:bg-blue-600 text-white hover:bg-blue-600 dark:hover:bg-blue-700 px-3 py-2 rounded-md float-right text-sm font-bold">
                {{ __('Add Package') }}
            </a>
        </x-slot>

        <div class="my-4">
            <div class="bg-white dark:bg-gray-800 shadow-sm sm:rounded-lg">
                <div class="px-4 py-5 sm:p-6">
                    <input type="text" wire:model.live.debounce.300ms="search"
                        class="block w-full max-w-xs rounded-md bg-white dark:bg-gray-700 px-3 py-1.5 text-base text-gray-900 dark:text-gray-100 outline-1 -outline-offset-1 outline-gray-300 dark:outline-gray-600 placeholder:text-gray-400 dark:placeholder:text-gray-500 focus:outline-2 focus:-outline-offset-2 focus:outline-indigo-600 sm:text-sm/6"
                        placeholder="{{ __('Search packages...') }}" />
                </div>
            </div>
        </div>

        {!! $this->renderNotification() !!}

        <div class="overflow-hidden shadow-sm ring-1 ring-black/5 dark:ring-white/10 sm:rounded-lg mt-2">
            <table class="min-w-full divide-y divide-gray-300 dark:divide-gray-600">
                <thead class="bg-gray-50 dark:bg-gray-700">
                    <tr>
                        <th scope="col" class="py-3.5 pl-4 pr-3 text-left text-sm font-semibold text-gray-900 dark:text-gray-100 sm:pl-3">#</th>
                        <th scope="col" class="px-3 py-3.5 text-left text-sm font-semibold text-gray-900 dark:text-gray-100">{{ __('Name') }}</th>
                        <th scope="col" class="px-3 py-3.5 text-left text-sm font-semibold text-gray-900 dark:text-gray-100">{{ __('Min / Max') }}</th>
                        <th scope="col" class="px-3 py-3.5 text-left text-sm font-semibold text-gray-900 dark:text-gray-100">{{ __('Interest %') }}</th>
                        <th scope="col" class="px-3 py-3.5 text-left text-sm font-semibold text-gray-900 dark:text-gray-100">{{ __('Duration (hours)') }}</th>
                        <th scope="col" class="px-3 py-3.5 text-left text-sm font-semibold text-gray-900 dark:text-gray-100">{{ __('Status') }}</th>
                        <th scope="col" class="relative py-3.5 pl-3 pr-4 sm:pr-3"><span class="sr-only">{{ __('Actions') }}</span></th>
                    </tr>
                </thead>
                <tbody class="bg-white dark:bg-gray-800">
                    @forelse ($this->results as $row)
                        <tr class="even:bg-gray-50 dark:even:bg-gray-800/50">
                            <td class="py-4 pl-4 pr-3 text-sm font-medium text-gray-900 dark:text-gray-100 sm:pl-3">{{ $row->id }}</td>
                            <td class="px-3 py-4 text-sm text-gray-900 dark:text-gray-100">{{ $row->name }}</td>
                            <td class="px-3 py-4 text-sm text-gray-500 dark:text-gray-400">{{ number_format((float) $row->min_amount, 2) }} / {{ number_format((float) $row->max_amount, 2) }}</td>
                            <td class="px-3 py-4 text-sm text-gray-500 dark:text-gray-400">{{ number_format((float) $row->interest_percent, 2) }}%</td>
                            <td class="px-3 py-4 text-sm text-gray-500 dark:text-gray-400">{{ $row->duration_hours }}</td>
                            <td class="px-3 py-4 text-sm">
                                @if ($row->is_active)
                                    <span class="px-2 inline-flex text-xs leading-5 font-semibold rounded-sm bg-green-400 text-white">{{ __('Active') }}</span>
                                @else
                                    <span class="px-2 inline-flex text-xs leading-5 font-semibold rounded-sm bg-red-400 text-white">{{ __('Inactive') }}</span>
                                @endif
                            </td>
                            <td class="relative py-4 pl-3 pr-4 text-right text-sm font-medium sm:pr-3">
                                <a href="{{ route('admin.investment-packages.edit', $row->id) }}" wire:navigate
                                    class="inline-flex items-center gap-x-1.5 rounded-md bg-indigo-600 px-2.5 py-1 text-xs font-semibold text-white shadow-xs hover:bg-indigo-500">
                                    {{ __('Edit') }}
                                </a>
                                @if ($row->is_active)
                                    <button type="button" wire:click="deactivate({{ $row->id }})"
                                        class="ml-1 inline-flex items-center rounded-md bg-yellow-500 px-2.5 py-1 text-xs font-semibold text-white hover:bg-yellow-600">
                                        {{ __('Deactivate') }}
                                    </button>
                                @else
                                    <button type="button" wire:click="activate({{ $row->id }})"
                                        class="ml-1 inline-flex items-center rounded-md bg-green-600 px-2.5 py-1 text-xs font-semibold text-white hover:bg-green-500">
                                        {{ __('Activate') }}
                                    </button>
                                @endif
                            </td>
                        </tr>
                    @empty
                        <tr>
                            <td colspan="7" class="py-6 text-center text-sm text-gray-500 dark:text-gray-400">{{ __('No packages found.') }}</td>
                        </tr>
                    @endforelse
                </tbody>
            </table>
        </div>

        <div class="mt-4">
            {{ $this->results->links() }}
        </div>
    </x-admin-panel>
</div>
