<?php

use App\Services\Trading\TradingSettingsService;
use App\Traits\Vrm\Livewire\WithNotifications;
use Livewire\Component;

new class extends Component {
    use WithNotifications;

    public bool $marketOpen = true;

    public string $mpesaWithdrawalMinAmount = '0';

    public string $mpesaWithdrawalMaxAmount = '0';

    public string $bankWithdrawalMinAmount = '0';

    public string $bankWithdrawalMaxAmount = '0';

    public string $sellMinSharesPerRequest = '1';

    public string $sellMaxSharesPerRequest = '0';

    public string $sellMaxSharesPerDay = '0';

    public string $depositMinAmount = '100';

    public string $depositMaxAmount = '0';

    public function mount(TradingSettingsService $settings): void
    {
        $this->marketOpen = $settings->isMarketOpen();
        $this->mpesaWithdrawalMinAmount = (string) $settings->withdrawalMinAmount('mpesa');
        $mpesaMax = $settings->withdrawalMaxAmount('mpesa');
        $this->mpesaWithdrawalMaxAmount = $mpesaMax >= PHP_FLOAT_MAX ? '0' : (string) $mpesaMax;

        $this->bankWithdrawalMinAmount = (string) $settings->withdrawalMinAmount('bank');
        $bankMax = $settings->withdrawalMaxAmount('bank');
        $this->bankWithdrawalMaxAmount = $bankMax >= PHP_FLOAT_MAX ? '0' : (string) $bankMax;

        $this->sellMinSharesPerRequest = (string) $settings->sellMinSharesPerRequest();
        $this->sellMaxSharesPerRequest = (string) $settings->sellMaxSharesPerRequest();
        $this->sellMaxSharesPerDay = (string) $settings->sellMaxSharesPerDay();

        $this->depositMinAmount = (string) $settings->depositMinAmount();
        $depositMax = $settings->depositMaxAmount();
        $this->depositMaxAmount = $depositMax >= PHP_FLOAT_MAX ? '0' : (string) $depositMax;
    }

    public function save(TradingSettingsService $settings): void
    {
        $validated = $this->validate([
            'marketOpen' => 'required|boolean',
            'mpesaWithdrawalMinAmount' => 'required|numeric|min:0',
            'mpesaWithdrawalMaxAmount' => 'required|numeric|min:0',
            'bankWithdrawalMinAmount' => 'required|numeric|min:0',
            'bankWithdrawalMaxAmount' => 'required|numeric|min:0',
            'sellMinSharesPerRequest' => 'required|integer|min:1',
            'sellMaxSharesPerRequest' => 'required|integer|min:0',
            'sellMaxSharesPerDay' => 'required|integer|min:0',
            'depositMinAmount' => 'required|numeric|min:0',
            'depositMaxAmount' => 'required|numeric|min:0',
        ]);

        if ((float) $validated['mpesaWithdrawalMaxAmount'] > 0 && (float) $validated['mpesaWithdrawalMaxAmount'] < (float) $validated['mpesaWithdrawalMinAmount']) {
            $this->addError('mpesaWithdrawalMaxAmount', __('M-Pesa withdrawal maximum amount must be greater than or equal to minimum amount.'));

            return;
        }

        if ((float) $validated['bankWithdrawalMaxAmount'] > 0 && (float) $validated['bankWithdrawalMaxAmount'] < (float) $validated['bankWithdrawalMinAmount']) {
            $this->addError('bankWithdrawalMaxAmount', __('Bank withdrawal maximum amount must be greater than or equal to minimum amount.'));

            return;
        }

        if ((int) $validated['sellMaxSharesPerRequest'] > 0 && (int) $validated['sellMaxSharesPerRequest'] < (int) $validated['sellMinSharesPerRequest']) {
            $this->addError('sellMaxSharesPerRequest', __('Sell maximum shares per request must be greater than or equal to minimum shares.'));

            return;
        }

        if ((int) $validated['sellMaxSharesPerDay'] > 0 && (int) $validated['sellMaxSharesPerDay'] < (int) $validated['sellMinSharesPerRequest']) {
            $this->addError('sellMaxSharesPerDay', __('Daily sell maximum must be greater than or equal to minimum shares per request.'));

            return;
        }

        if ((float) $validated['depositMaxAmount'] > 0 && (float) $validated['depositMaxAmount'] < (float) $validated['depositMinAmount']) {
            $this->addError('depositMaxAmount', __('Deposit maximum amount must be greater than or equal to minimum amount.'));

            return;
        }

        $settings->updateSettings(
            (bool) $validated['marketOpen'],
            $settings->investmentDurationDays(),
            $settings->investmentReturnPercent(),
        );
        $settings->updateWithdrawalLimits(
            (float) $validated['mpesaWithdrawalMinAmount'],
            (float) $validated['mpesaWithdrawalMaxAmount'],
            (float) $validated['bankWithdrawalMinAmount'],
            (float) $validated['bankWithdrawalMaxAmount'],
        );
        $settings->updateSellLimits(
            (int) $validated['sellMinSharesPerRequest'],
            (int) $validated['sellMaxSharesPerRequest'],
            (int) $validated['sellMaxSharesPerDay'],
        );
        $settings->updateDepositLimits(
            (float) $validated['depositMinAmount'],
            (float) $validated['depositMaxAmount'],
        );

        $this->notifySuccess(__('Settings updated successfully.'));
    }
}; ?>

<div>
    <x-admin-panel>
        <x-slot name="header">{{ __('Investment Settings') }}</x-slot>
        <x-slot name="desc">{{ __('Control market availability, withdrawal limits, and sell-share limits.') }}</x-slot>

        {!! $this->renderNotification() !!}

        <form wire:submit="save" class="space-y-6 max-w-2xl">
            <div class="overflow-hidden shadow-sm ring-1 ring-black/5 dark:ring-white/10 sm:rounded-lg">
                <div class="bg-white dark:bg-gray-800 px-4 py-5 sm:p-6 space-y-4">
                    <div>
                        <label for="marketOpen" class="required block text-sm/6 font-medium text-gray-900 dark:text-gray-100">{{ __('Market Status') }}</label>
                        <select id="marketOpen" wire:model="marketOpen"
                            class="mt-1 block w-full rounded-md bg-white dark:bg-gray-700 px-3 py-1.5 text-base text-gray-900 dark:text-gray-100 outline-1 -outline-offset-1 outline-gray-300 dark:outline-gray-600 focus:outline-2 focus:-outline-offset-2 focus:outline-indigo-600 sm:text-sm/6">
                            <option value="1">{{ __('Open') }}</option>
                            <option value="0">{{ __('Closed') }}</option>
                        </select>
                        <p class="mt-1 text-xs text-gray-500 dark:text-gray-400">{{ __('When closed, members cannot place trade orders.') }}</p>
                        @error('marketOpen')
                            <span class="text-red-500 text-xs">{{ $message }}</span>
                        @enderror
                    </div>

                    <div>
                        <p class="block text-sm/6 font-semibold text-gray-900 dark:text-gray-100">{{ __('M-Pesa Withdrawal Limits') }}</p>
                        <div class="mt-2 space-y-3">
                            <div>
                                <label for="mpesaWithdrawalMinAmount" class="required block text-sm/6 font-medium text-gray-900 dark:text-gray-100">{{ __('Minimum amount') }}</label>
                                <input id="mpesaWithdrawalMinAmount" type="number" step="0.01" min="0" wire:model="mpesaWithdrawalMinAmount"
                                    class="mt-1 block w-full rounded-md bg-white dark:bg-gray-700 px-3 py-1.5 text-base text-gray-900 dark:text-gray-100 outline-1 -outline-offset-1 outline-gray-300 dark:outline-gray-600 placeholder:text-gray-400 dark:placeholder:text-gray-500 focus:outline-2 focus:-outline-offset-2 focus:outline-indigo-600 sm:text-sm/6" />
                                @error('mpesaWithdrawalMinAmount')
                                    <span class="text-red-500 text-xs">{{ $message }}</span>
                                @enderror
                            </div>
                            <div>
                                <label for="mpesaWithdrawalMaxAmount" class="required block text-sm/6 font-medium text-gray-900 dark:text-gray-100">{{ __('Maximum amount') }}</label>
                                <input id="mpesaWithdrawalMaxAmount" type="number" step="0.01" min="0" wire:model="mpesaWithdrawalMaxAmount"
                                    class="mt-1 block w-full rounded-md bg-white dark:bg-gray-700 px-3 py-1.5 text-base text-gray-900 dark:text-gray-100 outline-1 -outline-offset-1 outline-gray-300 dark:outline-gray-600 placeholder:text-gray-400 dark:placeholder:text-gray-500 focus:outline-2 focus:-outline-offset-2 focus:outline-indigo-600 sm:text-sm/6" />
                                <p class="mt-1 text-xs text-gray-500 dark:text-gray-400">{{ __('Use 0 for no maximum.') }}</p>
                                @error('mpesaWithdrawalMaxAmount')
                                    <span class="text-red-500 text-xs">{{ $message }}</span>
                                @enderror
                            </div>
                        </div>
                    </div>

                    <div>
                        <p class="block text-sm/6 font-semibold text-gray-900 dark:text-gray-100">{{ __('Bank Withdrawal Limits') }}</p>
                        <div class="mt-2 space-y-3">
                            <div>
                                <label for="bankWithdrawalMinAmount" class="required block text-sm/6 font-medium text-gray-900 dark:text-gray-100">{{ __('Minimum amount') }}</label>
                                <input id="bankWithdrawalMinAmount" type="number" step="0.01" min="0" wire:model="bankWithdrawalMinAmount"
                                    class="mt-1 block w-full rounded-md bg-white dark:bg-gray-700 px-3 py-1.5 text-base text-gray-900 dark:text-gray-100 outline-1 -outline-offset-1 outline-gray-300 dark:outline-gray-600 placeholder:text-gray-400 dark:placeholder:text-gray-500 focus:outline-2 focus:-outline-offset-2 focus:outline-indigo-600 sm:text-sm/6" />
                                @error('bankWithdrawalMinAmount')
                                    <span class="text-red-500 text-xs">{{ $message }}</span>
                                @enderror
                            </div>
                            <div>
                                <label for="bankWithdrawalMaxAmount" class="required block text-sm/6 font-medium text-gray-900 dark:text-gray-100">{{ __('Maximum amount') }}</label>
                                <input id="bankWithdrawalMaxAmount" type="number" step="0.01" min="0" wire:model="bankWithdrawalMaxAmount"
                                    class="mt-1 block w-full rounded-md bg-white dark:bg-gray-700 px-3 py-1.5 text-base text-gray-900 dark:text-gray-100 outline-1 -outline-offset-1 outline-gray-300 dark:outline-gray-600 placeholder:text-gray-400 dark:placeholder:text-gray-500 focus:outline-2 focus:-outline-offset-2 focus:outline-indigo-600 sm:text-sm/6" />
                                <p class="mt-1 text-xs text-gray-500 dark:text-gray-400">{{ __('Use 0 for no maximum.') }}</p>
                                @error('bankWithdrawalMaxAmount')
                                    <span class="text-red-500 text-xs">{{ $message }}</span>
                                @enderror
                            </div>
                        </div>
                        <p class="mt-2 text-xs text-gray-500 dark:text-gray-400">{{ __('System minimum for any withdrawal is 0.') }}</p>
                    </div>

                    <div>
                        <p class="block text-sm/6 font-semibold text-gray-900 dark:text-gray-100">{{ __('Deposit Limits') }}</p>
                        <div class="mt-2 space-y-3">
                            <div>
                                <label for="depositMinAmount" class="required block text-sm/6 font-medium text-gray-900 dark:text-gray-100">{{ __('Minimum amount') }}</label>
                                <input id="depositMinAmount" type="number" step="0.01" min="0" wire:model="depositMinAmount"
                                    class="mt-1 block w-full rounded-md bg-white dark:bg-gray-700 px-3 py-1.5 text-base text-gray-900 dark:text-gray-100 outline-1 -outline-offset-1 outline-gray-300 dark:outline-gray-600 placeholder:text-gray-400 dark:placeholder:text-gray-500 focus:outline-2 focus:-outline-offset-2 focus:outline-indigo-600 sm:text-sm/6" />
                                <p class="mt-1 text-xs text-gray-500 dark:text-gray-400">{{ __('Minimum deposit amount for M-Pesa and other channels.') }}</p>
                                @error('depositMinAmount')
                                    <span class="text-red-500 text-xs">{{ $message }}</span>
                                @enderror
                            </div>
                            <div>
                                <label for="depositMaxAmount" class="required block text-sm/6 font-medium text-gray-900 dark:text-gray-100">{{ __('Maximum amount') }}</label>
                                <input id="depositMaxAmount" type="number" step="0.01" min="0" wire:model="depositMaxAmount"
                                    class="mt-1 block w-full rounded-md bg-white dark:bg-gray-700 px-3 py-1.5 text-base text-gray-900 dark:text-gray-100 outline-1 -outline-offset-1 outline-gray-300 dark:outline-gray-600 placeholder:text-gray-400 dark:placeholder:text-gray-500 focus:outline-2 focus:-outline-offset-2 focus:outline-indigo-600 sm:text-sm/6" />
                                <p class="mt-1 text-xs text-gray-500 dark:text-gray-400">{{ __('Use 0 for no maximum.') }}</p>
                                @error('depositMaxAmount')
                                    <span class="text-red-500 text-xs">{{ $message }}</span>
                                @enderror
                            </div>
                        </div>
                    </div>

                    <div>
                        <p class="block text-sm/6 font-semibold text-gray-900 dark:text-gray-100">{{ __('Sell Share Limits') }}</p>
                        <div class="mt-2 space-y-3">
                            <div>
                                <label for="sellMinSharesPerRequest" class="required block text-sm/6 font-medium text-gray-900 dark:text-gray-100">{{ __('Minimum shares per request') }}</label>
                                <input id="sellMinSharesPerRequest" type="number" step="1" min="1" wire:model="sellMinSharesPerRequest"
                                    class="mt-1 block w-full rounded-md bg-white dark:bg-gray-700 px-3 py-1.5 text-base text-gray-900 dark:text-gray-100 outline-1 -outline-offset-1 outline-gray-300 dark:outline-gray-600 placeholder:text-gray-400 dark:placeholder:text-gray-500 focus:outline-2 focus:-outline-offset-2 focus:outline-indigo-600 sm:text-sm/6" />
                                @error('sellMinSharesPerRequest')
                                    <span class="text-red-500 text-xs">{{ $message }}</span>
                                @enderror
                            </div>
                            <div>
                                <label for="sellMaxSharesPerRequest" class="required block text-sm/6 font-medium text-gray-900 dark:text-gray-100">{{ __('Maximum shares per request') }}</label>
                                <input id="sellMaxSharesPerRequest" type="number" step="1" min="0" wire:model="sellMaxSharesPerRequest"
                                    class="mt-1 block w-full rounded-md bg-white dark:bg-gray-700 px-3 py-1.5 text-base text-gray-900 dark:text-gray-100 outline-1 -outline-offset-1 outline-gray-300 dark:outline-gray-600 placeholder:text-gray-400 dark:placeholder:text-gray-500 focus:outline-2 focus:-outline-offset-2 focus:outline-indigo-600 sm:text-sm/6" />
                                <p class="mt-1 text-xs text-gray-500 dark:text-gray-400">{{ __('Use 0 for no maximum.') }}</p>
                                @error('sellMaxSharesPerRequest')
                                    <span class="text-red-500 text-xs">{{ $message }}</span>
                                @enderror
                            </div>
                            <div>
                                <label for="sellMaxSharesPerDay" class="required block text-sm/6 font-medium text-gray-900 dark:text-gray-100">{{ __('Maximum shares per day (per user)') }}</label>
                                <input id="sellMaxSharesPerDay" type="number" step="1" min="0" wire:model="sellMaxSharesPerDay"
                                    class="mt-1 block w-full rounded-md bg-white dark:bg-gray-700 px-3 py-1.5 text-base text-gray-900 dark:text-gray-100 outline-1 -outline-offset-1 outline-gray-300 dark:outline-gray-600 placeholder:text-gray-400 dark:placeholder:text-gray-500 focus:outline-2 focus:-outline-offset-2 focus:outline-indigo-600 sm:text-sm/6" />
                                <p class="mt-1 text-xs text-gray-500 dark:text-gray-400">{{ __('Use 0 for no daily limit.') }}</p>
                                @error('sellMaxSharesPerDay')
                                    <span class="text-red-500 text-xs">{{ $message }}</span>
                                @enderror
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <button type="submit"
                class="inline-flex items-center rounded-md bg-indigo-600 px-3 py-2 text-sm font-semibold text-white shadow-sm hover:bg-indigo-500">
                {{ __('Save Settings') }}
            </button>
        </form>
    </x-admin-panel>
</div>
