<?php

use App\Models\SecurityOrder;
use App\Models\User;
use App\Traits\Vrm\Livewire\WithNotifications;
use Livewire\Component;

new class extends Component {
    use WithNotifications;

    public ?User $member = null;
    public string $annualNetRatePercent = '0.0';

    public function mount(int $id): void
    {
        $this->member = User::query()
            ->with(['wallet', 'portfolioHoldings.security'])
            ->findOrFail($id);

        $storedRate = $this->member->getMeta('annual_net_rate_percent', 0);
        $this->annualNetRatePercent = is_numeric($storedRate) ? number_format((float) $storedRate, 1, '.', '') : '0.0';
    }

    public function activate(): void
    {
        if (! $this->member) {
            return;
        }

        $this->member->forceFill([
            'is_active' => true,
            'email_verified_at' => $this->member->email_verified_at ?? now(),
        ])->save();
        $this->member->refresh();

        $this->notifySuccess(__('Member account approved and activated.'));
    }

    public function suspend(): void
    {
        if (! $this->member) {
            return;
        }

        $this->member->update(['is_active' => false]);
        $this->member->refresh();

        $this->notifySuccess(__('Member account suspended.'));
    }

    public function updateAnnualNetRate(): void
    {
        if (! $this->member) {
            return;
        }

        $validated = $this->validate([
            'annualNetRatePercent' => ['required', 'numeric', 'min:0', 'max:1000'],
        ]);

        $rate = round((float) $validated['annualNetRatePercent'], 1);
        $this->member->setMeta('annual_net_rate_percent', $rate);
        $this->annualNetRatePercent = number_format($rate, 1, '.', '');

        $this->notifySuccess(__('Member annual net rate updated.'));
    }

    public function getPendingOrdersCountProperty(): int
    {
        if (! $this->member) {
            return 0;
        }

        return SecurityOrder::query()
            ->where('user_id', $this->member->id)
            ->where('status', 'pending')
            ->count();
    }
}; ?>

<div>
    <x-admin-panel>
        <x-slot name="header">{{ __('Edit Member') }}</x-slot>
        <x-slot name="desc">{{ __('Review profile, wallet, and portfolio details for this member.') }}</x-slot>
        <x-slot name="button">
            <a href="{{ route('admin.members.index') }}"
                class="bg-black dark:bg-gray-700 text-white hover:bg-gray-800 dark:hover:bg-gray-600 px-3 py-2 rounded-md float-right text-sm font-bold">
                {{ __('Go Back') }}
            </a>
        </x-slot>

        {!! $this->renderNotification() !!}

        @if ($member)
            <div class="space-y-4">
                <div class="rounded-lg border border-gray-200 bg-white p-4 dark:border-gray-700 dark:bg-gray-800">
                    <p class="text-lg font-semibold text-gray-900 dark:text-gray-100">{{ $member->name }}</p>
                    <p class="text-sm text-gray-500">{{ $member->email }}</p>
                    <p class="mt-2 text-sm text-gray-700 dark:text-gray-300">{{ __('Pending Orders') }}: {{ $this->pendingOrdersCount }}</p>
                    <div class="mt-4 flex gap-3">
                        @if ($member->is_active)
                            <button type="button" wire:click="suspend"
                                class="rounded-md bg-yellow-500 px-4 py-2 text-sm font-semibold text-white hover:bg-yellow-400">{{ __('Suspend Account') }}</button>
                        @else
                            <button type="button" wire:click="activate"
                                class="rounded-md bg-green-600 px-4 py-2 text-sm font-semibold text-white hover:bg-green-500">{{ __('Activate Account') }}</button>
                        @endif
                    </div>
                </div>

                <div class="rounded-lg border border-gray-200 bg-white p-4 dark:border-gray-700 dark:bg-gray-800">
                    <p class="text-sm font-semibold text-gray-900 dark:text-gray-100">{{ __('Wallet') }}</p>
                    <div class="mt-2 grid gap-2 text-sm text-gray-700 dark:text-gray-300">
                        <p>{{ __('Currency') }}: {{ $member->wallet ? app(\App\Services\DefaultCurrencyService::class)->code() : '-' }}</p>
                        <p>{{ __('Balance') }}: {{ number_format((float) ($member->wallet?->balance ?? 0), 2) }}</p>
                        <p>{{ __('Used') }}: {{ number_format((float) ($member->wallet?->used_amount ?? 0), 2) }}</p>
                        <p>{{ __('Earned') }}: {{ number_format((float) ($member->wallet?->earned_amount ?? 0), 2) }}</p>
                    </div>
                </div>

                <div class="rounded-lg border border-gray-200 bg-white p-4 dark:border-gray-700 dark:bg-gray-800">
                    <p class="text-sm font-semibold text-gray-900 dark:text-gray-100">{{ __('Dashboard Net Rate') }}</p>
                    <p class="mt-1 text-xs text-gray-500">{{ __('Shown to this member as 0.0% p.a on dashboard.') }}</p>
                    <div class="mt-3 flex flex-col gap-2 sm:flex-row sm:items-end">
                        <div class="w-full sm:max-w-xs">
                            <label for="annual-net-rate" class="mb-1 block text-xs font-medium text-gray-700 dark:text-gray-300">{{ __('Annual net rate (%)') }}</label>
                            <input id="annual-net-rate" type="number" step="0.1" min="0" max="1000" wire:model.live="annualNetRatePercent"
                                class="w-full rounded-md border border-gray-300 px-3 py-2 text-sm text-gray-900 focus:border-primary focus:outline-none dark:border-gray-600 dark:bg-gray-900 dark:text-gray-100">
                            @error('annualNetRatePercent')
                                <p class="mt-1 text-xs text-red-600">{{ $message }}</p>
                            @enderror
                        </div>
                        <button type="button" wire:click="updateAnnualNetRate"
                            class="rounded-md bg-black px-4 py-2 text-sm font-semibold text-white hover:bg-gray-800 dark:bg-gray-700 dark:hover:bg-gray-600">
                            {{ __('Save Rate') }}
                        </button>
                    </div>
                </div>

                <div class="rounded-lg border border-gray-200 bg-white p-4 dark:border-gray-700 dark:bg-gray-800">
                    <p class="text-sm font-semibold text-gray-900 dark:text-gray-100">{{ __('Portfolio Holdings') }}</p>
                    <div class="mt-3 space-y-2">
                        @forelse ($member->portfolioHoldings as $holding)
                            <div class="flex items-center justify-between rounded-md border border-gray-200 px-3 py-2 dark:border-gray-700">
                                <p class="text-sm text-gray-900 dark:text-gray-100">{{ $holding->security?->trading_name }}</p>
                                <p class="text-sm text-gray-600 dark:text-gray-300">{{ __('Qty') }}: {{ $holding->quantity }}</p>
                            </div>
                        @empty
                            <p class="text-sm text-gray-500">{{ __('No holdings yet.') }}</p>
                        @endforelse
                    </div>
                </div>
            </div>
        @endif
    </x-admin-panel>
</div>
