<?php

use App\Facades\Vrm\MediaForge;
use App\Models\Security;
use App\Traits\Vrm\Livewire\WithNotifications;
use Livewire\Attributes\Validate;
use Livewire\Component;
use Livewire\WithFileUploads;

new class extends Component {
    use WithFileUploads;
    use WithNotifications;

    #[Validate('required|string|max:20|unique:securities,trading_name')]
    public string $trading_name = '';

    #[Validate('nullable|string|max:255')]
    public string $name = '';

    #[Validate('required|string|min:3|max:4')]
    public string $currency_code = 'USD';

    #[Validate('required|numeric|min:0.0001')]
    public string $initial_listing_amount = '0.0001';

    #[Validate('required|numeric|min:0.0001')]
    public string $current_amount = '0.0001';

    #[Validate('required|integer|min:1')]
    public int $min_shares = 1;

    #[Validate('required|integer|min:0')]
    public int $max_shares_purchase = 0;

    #[Validate('required|integer|min:0')]
    public int $max_shares_holding = 0;

    #[Validate('nullable|image|mimes:jpg,jpeg,png,webp|max:2048')]
    public $logo;

    #[Validate('nullable|string|max:3000')]
    public string $description = '';

    public function save(): void
    {
        $this->validate();

        $logoPath = null;

        if ($this->logo) {
            $logoPath = MediaForge::upload($this->logo)
                ->useYearFolder(true)
                ->randomizeFileName(true)
                ->to('securities')
                ->resize(300, 300)
                ->run();
        }

        Security::query()->create([
            'name' => $this->name !== '' ? $this->name : strtoupper($this->trading_name),
            'trading_name' => strtoupper($this->trading_name),
            'logo' => $logoPath,
            'currency_code' => strtoupper($this->currency_code),
            'initial_listing_amount' => (float) $this->initial_listing_amount,
            'current_amount' => (float) $this->current_amount,
            'min_shares' => $this->min_shares,
            'max_shares_purchase' => $this->max_shares_purchase,
            'max_shares_holding' => $this->max_shares_holding,
            'description' => $this->description ?: null,
            'is_active' => true,
        ]);

        $this->notifySuccess(__('Security created successfully.'));

        $this->redirect(route('admin.securities.index'), navigate: true);
    }
}; ?>

<div>
    <x-admin-panel>
        <x-slot name="header">{{ __('Add Security') }}</x-slot>
        <x-slot name="desc">{{ __('Create a security with manual pricing, limits, and optional logo/description.') }}</x-slot>
        <x-slot name="button">
            <a href="{{ route('admin.securities.index') }}"
                class="bg-black dark:bg-gray-700 text-white hover:bg-gray-800 dark:hover:bg-gray-600 px-3 py-2 rounded-md float-right text-sm font-bold">
                {{ __('Go Back') }}
            </a>
        </x-slot>

        <div class="overflow-hidden shadow-sm ring-1 ring-black/5 dark:ring-white/10 sm:rounded-lg px-4 py-5 mb-5 sm:p-6">
            {!! $this->renderNotification() !!}

            <form wire:submit="save" class="space-y-6">
                <div class="grid grid-cols-1 gap-4 md:grid-cols-2">
                    <div>
                        <label class="required block text-sm font-medium text-gray-900 dark:text-gray-100">{{ __('Ticker Name') }}</label>
                        <input type="text" wire:model="trading_name"
                            class="mt-1 block w-full rounded-md bg-white dark:bg-gray-700 px-3 py-1.5 text-base text-gray-900 uppercase dark:text-gray-100 outline-1 -outline-offset-1 outline-gray-300 dark:outline-gray-600 placeholder:text-gray-400 dark:placeholder:text-gray-500 focus:outline-2 focus:-outline-offset-2 focus:outline-indigo-600 sm:text-sm/6" />
                        <span class="text-red-500 text-sm">{{ $errors->first('trading_name') }}</span>
                    </div>

                    <div>
                        <label class="block text-sm font-medium text-gray-900 dark:text-gray-100">{{ __('Display Name') }}</label>
                        <input type="text" wire:model="name"
                            class="mt-1 block w-full rounded-md bg-white dark:bg-gray-700 px-3 py-1.5 text-base text-gray-900 dark:text-gray-100 outline-1 -outline-offset-1 outline-gray-300 dark:outline-gray-600 focus:outline-2 focus:-outline-offset-2 focus:outline-indigo-600 sm:text-sm/6" />
                        <span class="text-red-500 text-sm">{{ $errors->first('name') }}</span>
                    </div>

                    <div>
                        <label class="block text-sm font-medium text-gray-900 dark:text-gray-100">{{ __('Logo') }}</label>
                        <input type="file" wire:model="logo" accept=".jpg,.jpeg,.png,.webp"
                            class="mt-1 block w-full cursor-pointer px-3 py-2 text-sm file:mr-4 file:rounded-md file:border-0 file:bg-gray-200 dark:file:bg-gray-600 file:px-3 file:py-1.5 file:text-sm file:font-semibold file:text-gray-900 dark:file:text-gray-100 hover:file:bg-gray-300 dark:hover:file:bg-gray-500 focus:outline-none focus:ring-2 focus:ring-gray-500 focus:ring-offset-2" />
                        <span class="text-red-500 text-sm">{{ $errors->first('logo') }}</span>
                        @if ($logo)
                            <img src="{{ $logo->temporaryUrl() }}" class="mt-2 h-16 w-16 rounded object-cover" alt="preview" />
                        @endif
                    </div>
                </div>

                <div class="grid grid-cols-1 gap-4 md:grid-cols-2">
                    <div>
                        <label class="required block text-sm font-medium text-gray-900 dark:text-gray-100">{{ __('Currency Code') }}</label>
                        <input type="text" wire:model="currency_code"
                            class="mt-1 block w-full rounded-md bg-white dark:bg-gray-700 px-3 py-1.5 text-base uppercase text-gray-900 dark:text-gray-100 outline-1 -outline-offset-1 outline-gray-300 dark:outline-gray-600 focus:outline-2 focus:-outline-offset-2 focus:outline-indigo-600 sm:text-sm/6" />
                        <span class="text-red-500 text-sm">{{ $errors->first('currency_code') }}</span>
                    </div>
                    <div>
                        <label class="required block text-sm font-medium text-gray-900 dark:text-gray-100">{{ __('Current Price') }}</label>
                        <input type="number" step="0.0001" min="0.0001" wire:model="current_amount"
                            class="mt-1 block w-full rounded-md bg-white dark:bg-gray-700 px-3 py-1.5 text-base text-gray-900 dark:text-gray-100 outline-1 -outline-offset-1 outline-gray-300 dark:outline-gray-600 focus:outline-2 focus:-outline-offset-2 focus:outline-indigo-600 sm:text-sm/6" />
                        <span class="text-red-500 text-sm">{{ $errors->first('current_amount') }}</span>
                    </div>
                    <div>
                        <label class="required block text-sm font-medium text-gray-900 dark:text-gray-100">{{ __('Initial Listing Price') }}</label>
                        <input type="number" step="0.0001" min="0.0001" wire:model="initial_listing_amount"
                            class="mt-1 block w-full rounded-md bg-white dark:bg-gray-700 px-3 py-1.5 text-base text-gray-900 dark:text-gray-100 outline-1 -outline-offset-1 outline-gray-300 dark:outline-gray-600 focus:outline-2 focus:-outline-offset-2 focus:outline-indigo-600 sm:text-sm/6" />
                        <span class="text-red-500 text-sm">{{ $errors->first('initial_listing_amount') }}</span>
                    </div>
                    <div>
                        <label class="required block text-sm font-medium text-gray-900 dark:text-gray-100">{{ __('Min Shares') }}</label>
                        <input type="number" step="1" min="1" wire:model="min_shares"
                            class="mt-1 block w-full rounded-md bg-white dark:bg-gray-700 px-3 py-1.5 text-base text-gray-900 dark:text-gray-100 outline-1 -outline-offset-1 outline-gray-300 dark:outline-gray-600 focus:outline-2 focus:-outline-offset-2 focus:outline-indigo-600 sm:text-sm/6" />
                        <span class="text-red-500 text-sm">{{ $errors->first('min_shares') }}</span>
                    </div>
                    <div>
                        <label class="required block text-sm font-medium text-gray-900 dark:text-gray-100">{{ __('Max Shares Per Purchase (0 = unlimited)') }}</label>
                        <input type="number" step="1" min="0" wire:model="max_shares_purchase"
                            class="mt-1 block w-full rounded-md bg-white dark:bg-gray-700 px-3 py-1.5 text-base text-gray-900 dark:text-gray-100 outline-1 -outline-offset-1 outline-gray-300 dark:outline-gray-600 focus:outline-2 focus:-outline-offset-2 focus:outline-indigo-600 sm:text-sm/6" />
                        <span class="text-red-500 text-sm">{{ $errors->first('max_shares_purchase') }}</span>
                    </div>
                    <div>
                        <label class="required block text-sm font-medium text-gray-900 dark:text-gray-100">{{ __('Max Shares Holding (0 = unlimited)') }}</label>
                        <input type="number" step="1" min="0" wire:model="max_shares_holding"
                            class="mt-1 block w-full rounded-md bg-white dark:bg-gray-700 px-3 py-1.5 text-base text-gray-900 dark:text-gray-100 outline-1 -outline-offset-1 outline-gray-300 dark:outline-gray-600 focus:outline-2 focus:-outline-offset-2 focus:outline-indigo-600 sm:text-sm/6" />
                        <span class="text-red-500 text-sm">{{ $errors->first('max_shares_holding') }}</span>
                    </div>
                </div>

                <div>
                    <label class="block text-sm font-medium text-gray-900 dark:text-gray-100">{{ __('Description') }}</label>
                    <textarea wire:model="description" rows="4"
                        class="mt-1 block w-full rounded-md bg-white dark:bg-gray-700 px-3 py-1.5 text-base text-gray-900 dark:text-gray-100 outline-1 -outline-offset-1 outline-gray-300 dark:outline-gray-600 placeholder:text-gray-400 dark:placeholder:text-gray-500 focus:outline-2 focus:-outline-offset-2 focus:outline-indigo-600 sm:text-sm/6"></textarea>
                    <span class="text-red-500 text-sm">{{ $errors->first('description') }}</span>
                </div>

                <div class="flex justify-end">
                    <button type="submit" class="rounded-md bg-indigo-600 px-4 py-2 text-sm font-semibold text-white hover:bg-indigo-500">
                        {{ __('Save Security') }}
                    </button>
                </div>
            </form>
        </div>
    </x-admin-panel>
</div>
