<?php

use App\Models\Security;
use App\Traits\Vrm\Livewire\WithNotifications;
use Livewire\Attributes\Computed;
use Livewire\Component;
use Livewire\WithPagination;

new class extends Component {
    use WithNotifications;
    use WithPagination;

    public string $search = '';
    public int $perPage = 10;

    public function updatedSearch(): void
    {
        $this->resetPage();
    }

    #[Computed]
    public function securities()
    {
        return Security::query()
            ->with('latestLog')
            ->with(['logs' => fn ($query) => $query->latest('logged_at')->limit(2)])
            ->withCount('logs')
            ->when($this->search !== '', function ($query): void {
                $query->where(function ($innerQuery): void {
                    $innerQuery->where('description', 'like', "%{$this->search}%")
                        ->orWhere('trading_name', 'like', "%{$this->search}%");
                });
            })
            ->orderByDesc('created_at')
            ->paginate($this->perPage);
    }

    public function activate(int $securityId): void
    {
        $security = Security::query()->find($securityId);

        if (! $security) {
            $this->notifyError(__('Security not found.'));

            return;
        }

        $security->update(['is_active' => true]);
        $this->notifySuccess(__('Security activated.'));
    }

    public function deactivate(int $securityId): void
    {
        $security = Security::query()->find($securityId);

        if (! $security) {
            $this->notifyError(__('Security not found.'));

            return;
        }

        $security->update(['is_active' => false]);
        $this->notifySuccess(__('Security deactivated.'));
    }

    public function delete(int $securityId): void
    {
        $security = Security::query()->find($securityId);

        if (! $security) {
            $this->notifyError(__('Security not found.'));

            return;
        }

        $security->delete();
        $this->notifySuccess(__('Security deleted.'));
    }

}; ?>

<div>
    <x-admin-panel>
        <x-slot name="header">{{ __('Securities') }}</x-slot>
        <x-slot name="desc">{{ __('Manage manual securities, latest prices, and history logs used by stock providers.') }}</x-slot>
        <x-slot name="button">
            <a href="{{ route('admin.securities.create') }}"
                class="bg-blue-500 dark:bg-blue-600 text-white hover:bg-blue-600 dark:hover:bg-blue-700 px-3 py-2 rounded-md float-right text-sm font-bold">
                {{ __('Add Security') }}
            </a>
        </x-slot>

        <div class="my-4">
            <div class="bg-white dark:bg-gray-800 shadow-sm sm:rounded-lg">
                <div class="px-4 py-5 sm:p-6">
                    <h3 class="text-base font-semibold text-gray-900 dark:text-gray-100">{{ __('Search securities') }}</h3>
                    <div class="mt-3 w-full sm:max-w-xs">
                        <input type="text" wire:model.live.debounce.300ms="search"
                            class="block w-full rounded-md bg-white dark:bg-gray-700 px-3 py-1.5 text-base text-gray-900 dark:text-gray-100 outline-1 -outline-offset-1 outline-gray-300 dark:outline-gray-600 placeholder:text-gray-400 dark:placeholder:text-gray-500 focus:outline-2 focus:-outline-offset-2 focus:outline-indigo-600 sm:text-sm/6"
                            placeholder="{{ __('Search by name or ticker...') }}" />
                    </div>
                </div>
            </div>
        </div>

        {!! $this->renderNotification() !!}

        <div class="overflow-hidden shadow-sm ring-1 ring-black/5 dark:ring-white/10 sm:rounded-lg mt-2">
            <table class="min-w-full divide-y divide-gray-300 dark:divide-gray-600">
                <thead class="bg-gray-50 dark:bg-gray-700">
                    <tr>
                        <th class="py-3.5 pr-3 pl-4 text-left text-sm font-semibold text-gray-900 dark:text-gray-100">{{ __('Logo') }}</th>
                        <th class="py-3.5 px-3 text-left text-sm font-semibold text-gray-900 dark:text-gray-100">{{ __('Ticker') }}</th>
                        <th class="py-3.5 px-3 text-left text-sm font-semibold text-gray-900 dark:text-gray-100">{{ __('Before') }}</th>
                        <th class="py-3.5 px-3 text-left text-sm font-semibold text-gray-900 dark:text-gray-100">{{ __('Now') }}</th>
                        <th class="py-3.5 px-3 text-left text-sm font-semibold text-gray-900 dark:text-gray-100">{{ __('Change %') }}</th>
                        <th class="py-3.5 px-3 text-left text-sm font-semibold text-gray-900 dark:text-gray-100">{{ __('Logs') }}</th>
                        <th class="py-3.5 px-3 text-left text-sm font-semibold text-gray-900 dark:text-gray-100">{{ __('Description') }}</th>
                        <th class="py-3.5 px-3 text-left text-sm font-semibold text-gray-900 dark:text-gray-100">{{ __('Status') }}</th>
                        <th class="py-3.5 pr-4 pl-3 text-right text-sm font-semibold text-gray-900 dark:text-gray-100">{{ __('Actions') }}</th>
                    </tr>
                </thead>
                <tbody class="bg-white dark:bg-gray-800">
                    @forelse ($this->securities as $security)
                        @php($beforeLog = $security->logs->get(1))
                        <tr class="even:bg-gray-50 dark:even:bg-gray-800/50">
                            <td class="py-4 pr-3 pl-4 text-sm text-gray-900 dark:text-gray-100">
                                @if ($security->logo)
                                    <img src="{{ asset($security->logo) }}" alt="{{ $security->trading_name }}" class="h-8 w-8 rounded object-cover" />
                                @else
                                    <span class="text-xs text-gray-500">{{ __('None') }}</span>
                                @endif
                            </td>
                            <td class="py-4 px-3 text-sm text-gray-900 dark:text-gray-100">{{ $security->trading_name }}</td>
                            <td class="py-4 px-3 text-sm text-gray-900 dark:text-gray-100">{{ $beforeLog ? number_format((float) $beforeLog->price, 4) : '-' }}</td>
                            <td class="py-4 px-3 text-sm text-gray-900 dark:text-gray-100">{{ number_format($security->latestPrice(), 4) }}</td>
                            <td class="py-4 px-3 text-sm text-gray-900 dark:text-gray-100">{{ number_format($security->latestChangePercent(), 4) }}</td>
                            <td class="py-4 px-3 text-sm text-gray-900 dark:text-gray-100">{{ $security->logs_count }}</td>
                            <td class="py-4 px-3 text-sm text-gray-900 dark:text-gray-100">{{ $security->description ?: '-' }}</td>
                            <td class="py-4 px-3 text-sm text-gray-900 dark:text-gray-100">
                                <span class="px-2 py-0.5 text-xs font-semibold rounded-sm {{ $security->is_active ? 'bg-green-500 text-white' : 'bg-red-500 text-white' }}">
                                    {{ $security->is_active ? __('Active') : __('Inactive') }}
                                </span>
                            </td>
                            <td class="py-4 pr-4 pl-3 text-right space-x-2">
                                <a href="{{ route('admin.securities.edit', $security->id) }}"
                                    class="inline-flex items-center rounded-md bg-indigo-600 px-2.5 py-1 text-xs font-semibold text-white hover:bg-indigo-500">{{ __('Edit / Logs') }}</a>

                                @if ($security->is_active)
                                    <button type="button" wire:click="deactivate({{ $security->id }})"
                                        class="inline-flex items-center rounded-md bg-yellow-500 px-2.5 py-1 text-xs font-semibold text-white hover:bg-yellow-400">{{ __('Hide') }}</button>
                                @else
                                    <button type="button" wire:click="activate({{ $security->id }})"
                                        class="inline-flex items-center rounded-md bg-emerald-600 px-2.5 py-1 text-xs font-semibold text-white hover:bg-emerald-500">{{ __('Show') }}</button>
                                @endif

                                <button type="button" wire:click="$js.confirmDelete({{ $security->id }})"
                                    class="inline-flex items-center rounded-md bg-red-600 px-2.5 py-1 text-xs font-semibold text-white hover:bg-red-500">{{ __('Delete') }}</button>
                            </td>
                        </tr>
                    @empty
                        <tr>
                            <td colspan="8" class="py-6 text-center text-sm text-gray-500 dark:text-gray-400">{{ __('No securities found.') }}</td>
                        </tr>
                    @endforelse
                </tbody>
            </table>
        </div>

        <div class="mt-8">{{ $this->securities->links() }}</div>
    </x-admin-panel>

    @script
        <script>
            $js('confirmDelete', async (securityId) => {
                const result = await Swal.fire({
                    title: 'Delete security?',
                    icon: 'warning',
                    showCancelButton: true,
                });

                if (result.isConfirmed) {
                    $wire.delete(securityId);
                }
            });
        </script>
    @endscript
</div>
