<?php

use App\Models\Investment;
use App\Models\InvestmentPackage;
use App\Models\WalletTransaction;
use App\Jobs\Vrm\SendMail;
use App\Mail\AdminTransactionMail;
use App\Mail\DepositRequestRejectedMail;
use App\Mail\WalletTransactionMail;
use App\Services\DefaultCurrencyService;
use App\Services\Investment\SubscriptionService;
use App\Services\MemberNotificationService;
use App\Services\Sms\SmsNotifier;
use App\Services\Vrm\InvestmentEmailService;
use App\Traits\Vrm\Livewire\WithNotifications;
use Illuminate\Support\Facades\DB;
use Livewire\Attributes\Computed;
use Livewire\Component;
use Livewire\WithPagination;

new class extends Component {
    use WithNotifications;
    use WithPagination;

    public string $status = 'pending';

    #[Computed]
    public function subscriptionRequests()
    {
        return WalletTransaction::query()
            ->with(['user', 'wallet'])
            ->whereIn('type', ['deposit_request', 'mpesa_deposit_request'])
            ->where('meta->intent', SubscriptionService::META_INTENT_SUBSCRIPTION)
            ->when($this->status !== 'all', fn ($query) => $query->where('status', $this->status))
            ->latest()
            ->paginate(12, ['*'], 'requests_page');
    }

    #[Computed]
    public function subscriptions()
    {
        return Investment::query()
            ->with(['user', 'package'])
            ->latest()
            ->paginate(12, ['*'], 'subscriptions_page');
    }

    public function approve(int $transactionId): void
    {
        $transaction = WalletTransaction::query()->with(['wallet', 'user'])->find($transactionId);

        if (! $transaction || $transaction->status !== 'pending') {
            $this->notifyError(__('Request not available for approval.'));

            return;
        }

        if (($transaction->meta['intent'] ?? null) !== SubscriptionService::META_INTENT_SUBSCRIPTION) {
            $this->notifyError(__('Request is not a subscription request.'));

            return;
        }

        $packageId = (int) ($transaction->meta['package_id'] ?? 0);
        $amount = (float) ($transaction->meta['amount'] ?? $transaction->amount);

        $package = InvestmentPackage::query()->find($packageId);

        if (! $package || ! $transaction->wallet || ! $transaction->user) {
            $this->notifyError(__('Subscription details are incomplete for this request.'));

            return;
        }

        $createdInvestment = null;

        DB::transaction(function () use ($transaction, $package, $amount, &$createdInvestment): void {
            $wallet = $transaction->wallet;

            $wallet->balance = (float) $wallet->balance + (float) $transaction->amount;
            $wallet->save();

            $transaction->status = 'approved';
            $transaction->approved_by = auth()->id();
            $transaction->approved_at = now();
            $transaction->save();

            $createdInvestment = app(SubscriptionService::class)->createSubscription($transaction->user, $wallet, $package, $amount);

            $meta = $transaction->meta ?? [];
            $meta['subscription_fulfilled'] = true;
            $transaction->meta = $meta;
            $transaction->save();
        });

        $transaction->refresh();

        app(MemberNotificationService::class)->notifySubscriptionReviewed(
            $transaction->user,
            $amount,
            app(DefaultCurrencyService::class)->code(),
            'approved',
            (int) $package->id,
        );

        if ($createdInvestment) {
            app(MemberNotificationService::class)->notifyInvestmentCreated($createdInvestment);
            app(MemberNotificationService::class)->notifyInvestmentReviewed($createdInvestment, 'approved');
        }

        if ($transaction->user?->email) {
            SendMail::dispatch($transaction->user->email, mailable: new WalletTransactionMail(
                $transaction->user->name,
                'deposit',
                $transaction->currency_code,
                (float) $transaction->amount,
                'approved',
                $transaction->phone_number ?: data_get($transaction->meta, 'phone_number', $transaction->user->phone),
                $transaction->transaction_code ?? data_get($transaction->meta, 'reference'),
                $transaction->updated_at?->format('M j, Y g:i A'),
            ), notificationType: 'user_transaction');
            app(InvestmentEmailService::class)->sendSubscriptionCreated(
                $transaction->user->email,
                $transaction->user->name,
                $package->name,
                app(DefaultCurrencyService::class)->code(),
                $amount,
            );
        }
        app(SmsNotifier::class)->sendToPhone(
            $transaction->phone_number ?: data_get($transaction->meta, 'phone_number', $transaction->user?->phone),
            "Deposit request approved. Subscription request approved for {$package->name}.",
            'user_transaction'
        );

        $adminEmail = config('mail.admin_address', 'admin@fxinvest.io');
        if (filled($adminEmail)) {
            SendMail::dispatch((string) $adminEmail, mailable: new AdminTransactionMail(
                $transaction->user?->name ?? 'Unknown',
                $transaction->user?->email ?? 'unknown@example.com',
                $amount,
                app(DefaultCurrencyService::class)->code(),
                'investment',
                'approved',
                'subscription',
                $transaction->transaction_code ?? data_get($transaction->meta, 'reference'),
            ), notificationType: 'admin_transaction');

            app(SmsNotifier::class)->sendToAdmins(
                "Subscription request approved for ".($transaction->user?->name ?? 'member').": ".app(DefaultCurrencyService::class)->code().' '.number_format((float) $amount, 2).'.',
                'admin_transaction'
            );
        }

        $this->notifySuccess(__('Subscription request approved.'));
    }

    public function reject(int $transactionId): void
    {
        $transaction = WalletTransaction::query()->with(['wallet', 'user'])->find($transactionId);

        if (! $transaction || $transaction->status !== 'pending') {
            $this->notifyError(__('Request not available for rejection.'));

            return;
        }

        if (($transaction->meta['intent'] ?? null) !== SubscriptionService::META_INTENT_SUBSCRIPTION) {
            $this->notifyError(__('Request is not a subscription request.'));

            return;
        }

        $transaction->status = 'rejected';
        $transaction->approved_by = auth()->id();
        $transaction->approved_at = now();
        $transaction->save();

        if ($transaction->user) {
            app(MemberNotificationService::class)->notifySubscriptionReviewed(
                $transaction->user,
                (float) (data_get($transaction->meta, 'amount') ?? $transaction->amount),
                app(DefaultCurrencyService::class)->code(),
                'rejected',
                (int) data_get($transaction->meta, 'package_id'),
            );
            app(MemberNotificationService::class)->notifyInvestmentReviewedForUser(
                $transaction->user,
                (float) (data_get($transaction->meta, 'amount') ?? $transaction->amount),
                app(DefaultCurrencyService::class)->code(),
                'rejected',
                (int) data_get($transaction->meta, 'package_id'),
            );
        }

        if ($transaction->user?->email) {
            SendMail::dispatch($transaction->user->email, mailable: new DepositRequestRejectedMail(
                $transaction->user->name,
                $transaction->currency_code,
                (float) $transaction->amount,
                $transaction->transaction_code ?? data_get($transaction->meta, 'reference'),
            ), notificationType: 'user_transaction');
        }
        app(SmsNotifier::class)->sendToPhone(
            $transaction->phone_number ?: data_get($transaction->meta, 'phone_number', $transaction->user?->phone),
            'Deposit request rejected. Subscription request rejected.',
            'user_transaction'
        );

        $adminEmail = config('mail.admin_address', 'admin@fxinvest.io');
        if (filled($adminEmail)) {
            SendMail::dispatch((string) $adminEmail, mailable: new AdminTransactionMail(
                $transaction->user?->name ?? 'Unknown',
                $transaction->user?->email ?? 'unknown@example.com',
                (float) (data_get($transaction->meta, 'amount') ?? $transaction->amount),
                app(DefaultCurrencyService::class)->code(),
                'investment',
                'rejected',
                'subscription',
                $transaction->transaction_code ?? data_get($transaction->meta, 'reference'),
            ), notificationType: 'admin_transaction');

            app(SmsNotifier::class)->sendToAdmins(
                "Subscription request rejected for ".($transaction->user?->name ?? 'member').": ".app(DefaultCurrencyService::class)->code().' '.number_format((float) (data_get($transaction->meta, 'amount') ?? $transaction->amount), 2).'.',
                'admin_transaction'
            );
        }

        $this->notifySuccess(__('Subscription request rejected.'));
    }
}; ?>

<div>
    <x-admin-panel>
        <x-slot name="header">{{ __('Subscriptions') }}</x-slot>
        <x-slot name="desc">{{ __('Approve or reject package subscription requests and review subscription records.') }}</x-slot>

        {!! $this->renderNotification() !!}

        <div class="mb-4">
            <label class="block text-sm/6 font-medium text-gray-900 dark:text-gray-100">{{ __('Request Status Filter') }}</label>
            <select wire:model.live="status"
                class="mt-1 block w-full max-w-xs rounded-md bg-white dark:bg-gray-700 px-3 py-1.5 text-base text-gray-900 dark:text-gray-100 outline-1 -outline-offset-1 outline-gray-300 dark:outline-gray-600 focus:outline-2 focus:-outline-offset-2 focus:outline-indigo-600 sm:text-sm/6">
                <option value="pending">{{ __('Pending') }}</option>
                <option value="approved">{{ __('Approved') }}</option>
                <option value="rejected">{{ __('Rejected') }}</option>
                <option value="all">{{ __('All') }}</option>
            </select>
        </div>

        <h3 class="text-base font-semibold text-gray-900 dark:text-gray-100 mb-2">{{ __('Subscription Requests') }}</h3>
        <div class="overflow-hidden shadow-sm ring-1 ring-black/5 dark:ring-white/10 sm:rounded-lg mt-2">
            <table class="min-w-full divide-y divide-gray-300 dark:divide-gray-600">
                <thead class="bg-gray-50 dark:bg-gray-700">
                    <tr>
                        <th class="py-3.5 pr-3 pl-4 text-left text-sm font-semibold text-gray-900 dark:text-gray-100">{{ __('Member') }}</th>
                        <th class="py-3.5 px-3 text-left text-sm font-semibold text-gray-900 dark:text-gray-100">{{ __('Package') }}</th>
                        <th class="py-3.5 px-3 text-left text-sm font-semibold text-gray-900 dark:text-gray-100">{{ __('Amount') }}</th>
                        <th class="py-3.5 px-3 text-left text-sm font-semibold text-gray-900 dark:text-gray-100">{{ __('Status') }}</th>
                        <th class="py-3.5 pr-4 pl-3 text-right text-sm font-semibold text-gray-900 dark:text-gray-100">{{ __('Actions') }}</th>
                    </tr>
                </thead>
                <tbody class="bg-white dark:bg-gray-800">
                    @forelse ($this->subscriptionRequests as $request)
                        <tr class="even:bg-gray-50 dark:even:bg-gray-800/50">
                            <td class="py-4 pr-3 pl-4 text-sm text-gray-900 dark:text-gray-100">{{ $request->user?->name ?? '-' }}</td>
                            <td class="py-4 px-3 text-sm text-gray-900 dark:text-gray-100">{{ data_get($request->meta, 'package_id') ?? '-' }}</td>
                            <td class="py-4 px-3 text-sm text-gray-900 dark:text-gray-100">{{ app(\App\Services\DefaultCurrencyService::class)->code() }} {{ number_format((float) (data_get($request->meta, 'amount') ?? $request->amount), 2) }}</td>
                            <td class="py-4 px-3 text-sm text-gray-900 dark:text-gray-100">{{ ucfirst($request->status) }}</td>
                            <td class="py-4 pr-4 pl-3 text-right">
                                @if ($request->status === 'pending')
                                    <button type="button" wire:click="$js.approveRequest({{ $request->id }})"
                                        class="inline-flex items-center rounded-md bg-green-600 px-2.5 py-1 text-xs font-semibold text-white hover:bg-green-500">{{ __('Approve') }}</button>
                                    <button type="button" wire:click="$js.rejectRequest({{ $request->id }})"
                                        class="inline-flex items-center rounded-md bg-red-600 px-2.5 py-1 text-xs font-semibold text-white hover:bg-red-500">{{ __('Reject') }}</button>
                                @else
                                    <span class="text-xs text-gray-500">{{ __('Processed') }}</span>
                                @endif
                            </td>
                        </tr>
                    @empty
                        <tr>
                            <td colspan="5" class="py-6 text-center text-sm text-gray-500 dark:text-gray-400">{{ __('No subscription requests found.') }}</td>
                        </tr>
                    @endforelse
                </tbody>
            </table>
        </div>
        <div class="mt-4">{{ $this->subscriptionRequests->links() }}</div>

        <h3 class="text-base font-semibold text-gray-900 dark:text-gray-100 mt-8 mb-2">{{ __('Subscriptions') }}</h3>
        <div class="overflow-hidden shadow-sm ring-1 ring-black/5 dark:ring-white/10 sm:rounded-lg mt-2">
            <table class="min-w-full divide-y divide-gray-300 dark:divide-gray-600">
                <thead class="bg-gray-50 dark:bg-gray-700">
                    <tr>
                        <th class="py-3.5 pr-3 pl-4 text-left text-sm font-semibold text-gray-900 dark:text-gray-100">{{ __('Member') }}</th>
                        <th class="py-3.5 px-3 text-left text-sm font-semibold text-gray-900 dark:text-gray-100">{{ __('Package') }}</th>
                        <th class="py-3.5 px-3 text-left text-sm font-semibold text-gray-900 dark:text-gray-100">{{ __('Principal') }}</th>
                        <th class="py-3.5 px-3 text-left text-sm font-semibold text-gray-900 dark:text-gray-100">{{ __('Expected Return') }}</th>
                        <th class="py-3.5 px-3 text-left text-sm font-semibold text-gray-900 dark:text-gray-100">{{ __('Status') }}</th>
                    </tr>
                </thead>
                <tbody class="bg-white dark:bg-gray-800">
                    @forelse ($this->subscriptions as $subscription)
                        <tr class="even:bg-gray-50 dark:even:bg-gray-800/50">
                            <td class="py-4 pr-3 pl-4 text-sm text-gray-900 dark:text-gray-100">{{ $subscription->user?->name ?? '-' }}</td>
                            <td class="py-4 px-3 text-sm text-gray-900 dark:text-gray-100">{{ $subscription->package?->name ?? '-' }}</td>
                            <td class="py-4 px-3 text-sm text-gray-900 dark:text-gray-100">{{ app(\App\Services\DefaultCurrencyService::class)->code() }} {{ number_format((float) $subscription->principal_amount, 2) }}</td>
                            <td class="py-4 px-3 text-sm text-gray-900 dark:text-gray-100">{{ app(\App\Services\DefaultCurrencyService::class)->code() }} {{ number_format((float) $subscription->expected_return_amount, 2) }}</td>
                            <td class="py-4 px-3 text-sm text-gray-900 dark:text-gray-100">{{ ucfirst($subscription->status) }}</td>
                        </tr>
                    @empty
                        <tr>
                            <td colspan="5" class="py-6 text-center text-sm text-gray-500 dark:text-gray-400">{{ __('No subscriptions found.') }}</td>
                        </tr>
                    @endforelse
                </tbody>
            </table>
        </div>
        <div class="mt-4">{{ $this->subscriptions->links() }}</div>
    </x-admin-panel>

    @script
        <script>
            $js('approveRequest', async (requestId) => {
                const result = await Swal.fire({
                    title: 'Approve subscription request?',
                    icon: 'question',
                    showCancelButton: true,
                });

                if (result.isConfirmed) {
                    $wire.approve(requestId);
                }
            });

            $js('rejectRequest', async (requestId) => {
                const result = await Swal.fire({
                    title: 'Reject subscription request?',
                    icon: 'warning',
                    showCancelButton: true,
                });

                if (result.isConfirmed) {
                    $wire.reject(requestId);
                }
            });
        </script>
    @endscript
</div>
