<?php

use App\Models\Security;
use App\Models\SecurityOrder;
use App\Models\WalletTransaction;
use App\Services\DefaultCurrencyService;
use App\Services\Trading\TradingSettingsService;
use App\Models\Wallet;
use Illuminate\Database\Eloquent\Collection;
use Livewire\Attributes\Computed;
use Livewire\Attributes\Layout;
use Livewire\Component;

new #[Layout('layouts.trading')] class extends Component {
    public bool $showBalance = true;

    public function toggleBalanceVisibility(): void
    {
        $this->showBalance = !$this->showBalance;
    }

    #[Computed]
    public function wallet(): Wallet
    {
        return Wallet::query()->firstOrCreate(['user_id' => auth()->id()], ['currency_code' => app(DefaultCurrencyService::class)->code()]);
    }

    #[Computed]
    public function securities(): array
    {
        return Security::query()->where('is_active', true)->with('latestLog')->orderBy('trading_name')->get()->map(fn(Security $security): array => $this->mapSecurityForDisplay($security))->values()->all();
    }

    #[Computed]
    public function holdings(): array
    {
        return auth()
            ->user()
            ->portfolioHoldings()
            ->with('security')
            ->get()
            ->map(function ($holding): array {
                $latestPrice = (float) $holding->security->latestPrice();
                $currentValue = (float) $holding->quantity * $latestPrice;
                $gainLossPercent = (float) $holding->average_buy_price > 0 ? (($latestPrice - (float) $holding->average_buy_price) / (float) $holding->average_buy_price) * 100 : 0;

                return [
                    'symbol' => $holding->security->trading_name,
                    'name' => $holding->security->name,
                    'current_value' => $currentValue,
                    'gain_loss_percent' => $gainLossPercent,
                ];
            })
            ->all();
    }

    #[Computed]
    public function portfolio(): array
    {
        $inTrade = collect($this->holdings)->sum('current_value');
        $wallet = $this->wallet;
        $invested = auth()->user()->portfolioHoldings()->sum('total_invested');
        $gainLoss = $invested > 0 ? (($inTrade - (float) $invested) / (float) $invested) * 100 : 0;

        return [
            'balance' => (float) $wallet->balance,
            'inTrade' => (float) $inTrade,
            'gainLoss' => (float) $gainLoss,
            'currency' => app(DefaultCurrencyService::class)->code(),
        ];
    }

    #[Computed]
    public function watchlist(): array
    {
        return auth()
            ->user()
            ->watchlistItems()
            ->with('security')
            ->latest()
            ->limit(5)
            ->get()
            ->map(
                fn($item): array => [
                    'id' => $item->security->id,
                    'symbol' => $item->security->trading_name,
                    'name' => $item->security->name,
                    'price' => (float) $item->security->latestPrice(),
                    'change' => $item->security->latestChangePercent(),
                ],
            )
            ->all();
    }

    #[Computed]
    public function newestSecurity(): ?array
    {
        $security = Security::query()->latest()->first();

        if (!$security) {
            return null;
        }

        return [
            'name' => $security->name,
            'symbol' => $security->trading_name,
        ];
    }

    /** Pending orders (trades) + pending wallet transactions (deposits/withdrawals). */
    #[Computed]
    public function pendingTransactionsCount(): int
    {
        $orders = SecurityOrder::query()
            ->where('user_id', auth()->id())
            ->where('status', 'pending')
            ->count();

        $walletRequests = WalletTransaction::query()
            ->where('user_id', auth()->id())
            ->where('status', 'pending')
            ->count();

        return $orders + $walletRequests;
    }

    #[Computed]
    public function isMarketOpen(): bool
    {
        return app(TradingSettingsService::class)->isMarketOpen();
    }

    /** Default currency from ATU Currency (used for display where wallet is not in scope, e.g. marquee). */
    #[Computed]
    public function defaultCurrencyCode(): string
    {
        return app(DefaultCurrencyService::class)->code();
    }

    #[Computed]
    public function recentTransactions(): Collection
    {
        return WalletTransaction::query()
            ->where('user_id', auth()->id())
            ->latest()
            ->limit(10)
            ->get();
    }

    #[Computed]
    public function annualNetRatePercent(): float
    {
        $rawValue = auth()->user()?->getMeta('annual_net_rate_percent', 0);

        if (!is_numeric($rawValue)) {
            return 0.0;
        }

        return max(0, (float) $rawValue);
    }

    private function mapSecurityForDisplay(Security $security): array
    {
        return [
            'id' => $security->id,
            'symbol' => strtoupper((string) $security->trading_name),
            'name' => (string) ($security->name ?: $security->trading_name),
            'price' => $security->latestPrice(),
            'change' => $security->latestChangePercent(),
            'logo' => $security->logo,
            'initials' => $this->buildInitials((string) $security->trading_name, (string) ($security->name ?: $security->trading_name)),
        ];
    }

    private function buildInitials(string $symbol, string $name): string
    {
        $lettersOnly = preg_replace('/[^A-Z]/', '', strtoupper($symbol)) ?? '';
        if ($lettersOnly !== '') {
            return substr($lettersOnly, 0, 2);
        }

        $nameParts = preg_split('/\s+/', trim($name)) ?: [];
        $initials = collect($nameParts)->filter()->take(2)->map(fn(string $part): string => strtoupper(substr($part, 0, 1)))->implode('');

        return $initials !== '' ? $initials : 'ST';
    }
}; ?>

<div>
	@include('partials.trading.topbar', ['title' => 'Ziidi'])
	@include('partials.trading.marquee', [
		'securities' => $this->securities,
		'currencyCode' => $this->defaultCurrencyCode,
	])

	<div class="px-4 py-4 space-y-5">
		<section class="dashboard-balance-card rounded-[2rem] p-5 text-white shadow-sm">
			<p class="text-center text-sm text-white/90 sm:text-base">{{ __('Investment balance') }}</p>
			<div class="mt-2 flex items-center justify-center gap-3">
				<h2 class="text-xl font-extrabold tracking-tight sm:text-3xl">
					@if ($showBalance)
						{{ $this->portfolio['currency'] }} {{ number_format($this->portfolio['balance'], 2) }}
					@else
						{{ __('******') }}
					@endif
				</h2>
				<button type="button" wire:click="toggleBalanceVisibility"
					class="inline-flex h-10 w-10 items-center justify-center rounded-full  text-white transition hover:bg-white/10"
					aria-label="{{ $showBalance ? __('Hide balance') : __('Show balance') }}">
					<i class="fa-solid {{ $showBalance ? 'fa-eye-slash' : 'fa-eye' }} text-base sm:text-lg"></i>
				</button>
			</div>

			<div class="mt-5 grid grid-cols-2 gap-3 text-center">
				<div>
					<p class="text-xs text-white/90 sm:text-sm">{{ __('Interest earned yesterday') }}</p>
					<p class="text-lg font-bold lowercase sm:text-xl">{{ __('n/a') }}</p>
				</div>
				<div>
					<p class="text-xs text-white/90 sm:text-sm">{{ __('Net rate (last 24 hrs)') }}</p>
					<p class="text-lg font-bold sm:text-xl">
						{{ number_format($this->annualNetRatePercent, 1) }}% p.a
					</p>
				</div>
			</div>

			<div class="mt-6 grid grid-cols-2 gap-4">
				<a href="{{ route($this->portfolio['balance'] > 0 ? 'member.investments' : 'member.mpesa-deposit') }}"
					wire:navigate
					class="inline-flex items-center justify-center gap-2 rounded-full bg-white px-4 py-2 text-sm font-bold text-brand-green transition hover:bg-white/90">
					<img src="{{ asset('media/images/invenst-icon-1.png') }}" alt="{{ __('Invest') }}"
						class="h-4 w-4 object-contain" />
					<span>{{ __('Invest') }}</span>
				</a>
				<a href="{{ route('member.withdraw') }}" wire:navigate
					class="inline-flex items-center justify-center gap-2 rounded-full border border-white/60 bg-transparent px-4 py-2 text-sm font-bold text-white transition hover:bg-white/10">
					<i class="fa-solid fa-hand-holding-dollar text-sm"></i>
					<span>{{ __('Withdraw') }}</span>
				</a>
			</div>
		</section>

		<div
			class="{{ $this->isMarketOpen ? 'border-primary/30 bg-primary/10' : 'border-destructive/30 bg-destructive/10' }} rounded-xl border px-4 py-3">
			<div class="flex items-center gap-2">
				<i class="fa-solid fa-circle-info h-4 w-4 {{ $this->isMarketOpen ? 'text-primary' : 'text-destructive' }}"></i>
				<span class="text-xs font-medium sm:text-sm {{ $this->isMarketOpen ? 'text-primary' : 'text-destructive' }}">
					{{ $this->isMarketOpen ? __('Market is open') : __('Market is closed for today.') }}
				</span>
			</div>
		</div>

		<section class="grid grid-cols-2 gap-3">
			<a href="{{ route('member.investments') }}" wire:navigate
				class="overflow-hidden rounded-2xl border border-[#F4D8AF]">
				<div class="bg-[#FFEACE] px-4 py-4 text-center">
					<img src="{{ asset('media/images/pad-lock-zidii.png') }}" alt="{{ __('Zidii Trade') }}"
						class="mx-auto h-10 w-10 object-contain">
				</div>
				<div class="bg-[#FFF4E6] px-4 py-3 text-center">
					<p class="text-sm font-semibold text-[#71706f] sm:text-base">{{ __('ZiiDi Lock') }}</p>
					<p class="text-[11px] text-muted-foreground sm:text-xs">{{ __('Feeling spontaneous with your wealth? Lock it!') }}
					</p>
				</div>
			</a>
			<a href="{{ route('member.investments') }}" wire:navigate
				class="overflow-hidden rounded-2xl border border-[#BCEAD0]">
				<div class="bg-[#CDF3DC] px-4 py-4 text-center">
					<img src="{{ asset('media/images/cash-bag-zidii.png') }}" alt="{{ __('Auto Invest') }}"
						class="mx-auto h-10 w-10 object-contain">
				</div>
				<div class="bg-[#EAF9F0] px-4 py-3 text-center">
					<p class="text-sm font-semibold text-[#71706f] sm:text-base">{{ __('Auto Invest') }}</p>
					<p class="text-[11px] text-muted-foreground sm:text-xs">{{ __('Spend like a pro, invest like a champion') }}</p>
				</div>
			</a>
		</section>

		<section class="space-y-2">
			<p class="text-xs font-semibold text-foreground sm:text-sm">{{ __('Transaction') }}</p>
			@forelse ($this->recentTransactions as $transaction)
				<div class="flex items-center justify-between rounded-xl border border-border bg-card p-3">
					<div>
						<p class="text-xs font-semibold text-foreground sm:text-sm">
							{{ ucfirst(str_replace('_', ' ', $transaction->type)) }}</p>
						<p class="text-[11px] text-muted-foreground sm:text-xs">{{ $transaction->created_at?->format('Y-m-d H:i') }}</p>
					</div>
					<div class="text-right">
						<p class="text-xs font-semibold text-foreground sm:text-sm">{{ $this->portfolio['currency'] }}
							{{ number_format((float) $transaction->amount, 2) }}</p>
						@php($normalizedStatus = strtolower((string) $transaction->status))
						<p
							class="text-[11px] text-muted-foreground transaction-status sm:text-xs {{ $normalizedStatus === 'pending' ? 'transaction-status--pending' : ($normalizedStatus === 'rejected' ? 'transaction-status--rejected' : (in_array($normalizedStatus, ['approved', 'completed'], true) ? 'transaction-status--approved' : '')) }}">
							{{ ucfirst($transaction->status) }}
						</p>
					</div>
				</div>
			@empty
				<p class="text-xs text-muted-foreground sm:text-sm">{{ __('No transactions yet.') }}</p>
			@endforelse

			<div class="pt-1">
				<a href="{{ route('member.transactions') }}" wire:navigate
					class="inline-flex items-center rounded-lg bg-[#2CB34A] px-3 py-2 text-xs font-semibold text-white transition hover:bg-[#24933d] sm:text-sm">
					{{ __('Show more') }}
				</a>
			</div>
		</section>
	</div>
</div>
