<?php

use App\Models\Wallet;
use App\Models\WalletTransaction;
use App\Jobs\Vrm\SendMail;
use App\Mail\AdminDepositMail;
use App\Mail\WalletTransactionMail;
use App\Services\DefaultCurrencyService;
use App\Services\MemberNotificationService;
use App\Services\Investment\SubscriptionService;
use App\Services\Trading\TradingSettingsService;
use App\Services\Sms\SmsNotifier;
use App\Traits\Vrm\Livewire\WithNotifications;
use Illuminate\Support\Sleep;
use Livewire\Attributes\Computed;
use Livewire\Attributes\Layout;
use Livewire\Component;

new #[Layout('layouts.trading')] class extends Component {
    use WithNotifications;

    public string $amount = '';
    public string $phoneNumber = '';
    public string $depositStep = 'form';
    public float $submittedDepositAmount = 0.0;
    public string $intent = '';
    public string $intentPackageId = '';
    public string $targetAmount = '';

    public function mount(): void
    {
        $this->useToasts();

        $amountFromQuery = request()->query('amount');
        if (is_string($amountFromQuery) && is_numeric($amountFromQuery) && (float) $amountFromQuery > 0) {
            $this->amount = number_format((float) $amountFromQuery, 2, '.', '');
        }

        $intentFromQuery = (string) request()->query('intent', '');
        if ($intentFromQuery === SubscriptionService::META_INTENT_SUBSCRIPTION) {
            $this->intent = $intentFromQuery;
            $this->intentPackageId = (string) request()->query('package_id', '');
            $this->targetAmount = (string) request()->query('target_amount', '');
        }
    }

    #[Computed]
    public function wallet(): Wallet
    {
        return Wallet::query()->firstOrCreate(['user_id' => auth()->id()], ['currency_code' => app(DefaultCurrencyService::class)->code()]);
    }

    /** System default currency from ATU (single currency across the app). */
    #[Computed]
    public function currencyCode(): string
    {
        return app(DefaultCurrencyService::class)->code();
    }

    #[Computed]
    public function depositLimits(): array
    {
        return app(TradingSettingsService::class)->depositLimits();
    }

    #[Computed]
    public function requests()
    {
        return WalletTransaction::query()
            ->where('user_id', auth()->id())
            ->where('type', 'mpesa_deposit_request')
            ->latest()
            ->limit(10)
            ->get();
    }

    public function submitDepositRequest(): void
    {
        $limits = $this->depositLimits;
        $rules = [
            'amount' => ['required', 'numeric', 'min:' . $limits['min']],
            'phoneNumber' => ['required', 'string', 'size:12', 'regex:/^254(?:7|1)\d{8}$/'],
        ];
        if ($limits['max'] < PHP_FLOAT_MAX) {
            $rules['amount'][] = 'max:' . $limits['max'];
        }
        $this->validate($rules, [
            'amount.min' => __('The deposit amount must be at least :min.', ['min' => number_format($limits['min'], 2)]),
            'amount.max' => __('The deposit amount may not be greater than :max.', ['max' => number_format($limits['max'], 2)]),
        ]);

        $wallet = $this->wallet;
        $mpesaService = app(\App\Services\Payments\MpesaService::class);
        $normalizedPhone = $mpesaService->normalizePhoneNumber('+' . $this->phoneNumber);

        // Initiate STK Push
        $response = $mpesaService->stkPush(
            $normalizedPhone,
            (float) $this->amount,
            'Deposit-' . auth()->id(),
            'Wallet Deposit'
        );

        if (!$response['status']) {
            $this->notifyError($response['message'] ?: __('M-Pesa request failed. Please try again.'));
            return;
        }

        $stkData = $response['data'];

        $meta = [
            'channel' => 'mpesa',
            'phone_number' => $this->phoneNumber,
            'merchant_request_id' => $stkData['MerchantRequestID'] ?? null,
            'checkout_request_id' => $stkData['CheckoutRequestID'] ?? null,
        ];

        if ($this->intent === SubscriptionService::META_INTENT_SUBSCRIPTION && is_numeric($this->intentPackageId)) {
            $meta['intent'] = SubscriptionService::META_INTENT_SUBSCRIPTION;
            $meta['package_id'] = (int) $this->intentPackageId;
            $meta['amount'] = is_numeric($this->targetAmount) && (float) $this->targetAmount > 0 ? (float) $this->targetAmount : (float) $this->amount;
        }

        $transaction = WalletTransaction::query()->create([
            'wallet_id' => $wallet->id,
            'user_id' => auth()->id(),
            'type' => 'mpesa_deposit_request',
            'provider' => 'mpesa',
            'status' => 'pending',
            'amount' => (float) $this->amount,
            'currency_code' => app(DefaultCurrencyService::class)->code(),
            'description' => 'M-Pesa wallet deposit request',
            'phone_number' => $this->phoneNumber,
            'merchant_request_id' => $stkData['MerchantRequestID'] ?? null,
            'checkout_request_id' => $stkData['CheckoutRequestID'] ?? null,
            'transaction_code' => null,
            'meta' => $meta,
        ]);

        app(MemberNotificationService::class)->notifyDepositRequested($transaction);

        if (($meta['intent'] ?? null) === SubscriptionService::META_INTENT_SUBSCRIPTION) {
            app(MemberNotificationService::class)->notifySubscriptionRequested(
                auth()->user(),
                (float) ($meta['amount'] ?? $transaction->amount),
                app(DefaultCurrencyService::class)->code(),
                isset($meta['package_id']) ? (int) $meta['package_id'] : null,
            );
        }

        $adminEmail = config('mail.admin_address');
        app(SmsNotifier::class)->sendToPhone(
            $transaction->phone_number ?: auth()->user()->phone,
            "Deposit request submitted: {$transaction->currency_code} ".number_format((float) $transaction->amount, 2).'. Please enter your PIN on your phone.',
            'user_transaction'
        );

        app(SmsNotifier::class)->sendToAdmins(
            "Deposit request submitted by ".auth()->user()->name.": {$transaction->currency_code} ".number_format((float) $transaction->amount, 2).'.',
            'admin_deposit'
        );

        if (filled($adminEmail)) {
            try {
                SendMail::dispatch((string) $adminEmail, mailable: new AdminDepositMail(
                    auth()->user()->name,
                    auth()->user()->email,
                    (float) $transaction->amount,
                    $transaction->currency_code,
                    'submitted',
                    'mpesa',
                    $transaction->transaction_code,
                ), notificationType: 'admin_deposit');
            } catch (\Throwable $exception) {
                \Illuminate\Support\Facades\Log::warning('Admin deposit email dispatch failed.', [
                    'transaction_id' => $transaction->id,
                    'message' => $exception->getMessage(),
                ]);
            }
        }

        try {
            SendMail::dispatch((string) auth()->user()->email, mailable: new WalletTransactionMail(
                auth()->user()->name,
                'mpesa_deposit_request',
                $transaction->currency_code,
                (float) $transaction->amount,
                'pending',
                $transaction->phone_number,
                $transaction->transaction_code,
                $transaction->created_at?->format('M j, Y g:i A'),
            ), notificationType: 'user_transaction');
        } catch (\Throwable $exception) {
            \Illuminate\Support\Facades\Log::warning('Member deposit email dispatch failed.', [
                'transaction_id' => $transaction->id,
                'message' => $exception->getMessage(),
            ]);
        }

        $this->submittedDepositAmount = (float) $this->amount;
        $this->depositStep = 'result';
        $this->intent = '';
        $this->intentPackageId = '';
        $this->targetAmount = '';

        $this->notifySuccess(__('STK Push sent to your phone.'));
    }

    public function resetDepositFlow(): void
    {
        $this->depositStep = 'form';
        $this->amount = '';
        $this->phoneNumber = '';
        $this->submittedDepositAmount = 0.0;
    }

    public function goToDashboard(): void
    {
        $this->redirectRoute('member.dashboard', navigate: true);
    }

    #[Computed]
    public function maskedPhoneNumber(): string
    {
        $digits = preg_replace('/\D+/', '', $this->phoneNumber);

        if (str_starts_with($digits, '254')) {
            $digits = '0' . substr($digits, 3);
        }

        if (str_starts_with($digits, '07') || str_starts_with($digits, '01')) {
            return substr($digits, 0, 2) . '****';
        }

        return '01****';
    }

    public function completePayment(): void
    {
        $this->redirectRoute('member.transactions', navigate: true);
    }
}; ?>

<div>
	<div x-data="{ scrolled: false }" x-init="scrolled = window.scrollY > 6;
window.addEventListener('scroll', () => scrolled = window.scrollY > 6)"
		:class="scrolled ? 'bg-zinc-950 border-zinc-800' : 'bg-background border-border'"
		class="sticky top-0 z-40 border-b px-4 py-3 transition-colors duration-200">
		<div class="flex items-start justify-between gap-3">
			<a href="{{ route('member.dashboard') }}" wire:navigate class="mt-1 text-foreground">
				<i class="fa-solid fa-arrow-left text-base"></i>
			</a>
			<div class="flex-1">
				<p class="text-base font-semibold text-foreground">{{ __('Deposit Funds') }}</p>
				<p class="text-sm text-muted-foreground">{{ __('Add funds to start investing') }}</p>
			</div>
			<button type="button" class="mt-1 text-foreground" aria-label="{{ __('Help') }}">
				<i class="fa-regular fa-circle-question text-base"></i>
			</button>
		</div>
	</div>

	<div class="mx-auto w-full max-w-xl px-4 py-5 space-y-4">
		@if (!empty($this->notification))
			<div x-data="{ show: true }" x-show="show" x-transition:enter="transform transition ease-out duration-300"
				x-transition:enter-start="translate-y-3 opacity-0" x-transition:enter-end="translate-y-0 opacity-100"
				x-transition:leave="transform transition ease-in duration-200"
				x-transition:leave-start="translate-y-0 opacity-100" x-transition:leave-end="translate-y-3 opacity-0"
				x-init="setTimeout(() => {
    show = false;
    $wire.clearNotification();
}, 3500)"
				class="fixed inset-x-0 bottom-24 z-[96] px-4 lg:bottom-6 [&_.toast]:mx-auto [&_.toast]:max-w-sm [&_.toast]:w-full">
				{!! $this->renderNotification() !!}
			</div>
		@endif

		@if ($depositStep === 'form')
			<div class="bg-card rounded-2xl border border-border p-4 sm:p-5 space-y-4">
				<div class="rounded-xl border border-emerald-500/60 bg-primary/10 p-3.5">
					<div class="flex items-center gap-3">
						<div class="size-11 rounded-xl bg-emerald-600 text-white flex items-center justify-center text-md font-bold">M
						</div>
						<div>
							<p class="text-sm font-semibold text-foreground dark:text-white">{{ __('M-Pesa Instant Deposit') }}</p>
							<p class="text-xs text-muted-foreground dark:text-white/90">{{ __('Fast & secure mobile money transfer') }}</p>
						</div>
					</div>
				</div>

				@if ($intent === \App\Services\Investment\SubscriptionService::META_INTENT_SUBSCRIPTION)
					<div class="rounded-xl border border-primary/30 bg-primary/10 px-3 py-2 text-xs text-primary">
						{{ __('Top up this amount to complete your package subscription after admin approval.') }}
					</div>
				@endif

				<div class="space-y-2">
					<label class="block text-sm font-medium text-foreground">{{ __('Phone Number') }}</label>
					<div class="relative">
						<i class="fa-solid fa-phone absolute left-4 top-1/2 -translate-y-1/2 text-sm text-muted-foreground"></i>
						<input type="text" wire:model="phoneNumber"
							maxlength="12"
							class="w-full rounded-xl border border-border bg-background py-2.5 pl-11 pr-4 text-sm text-foreground"
							placeholder="254712345678" />
					</div>
					<p class="text-xs text-muted-foreground">{{ __('Enter your M-Pesa registered phone number starting with 254 (e.g. 254712345678).') }}
					</p>
					<span class="text-destructive text-xs">{{ $errors->first('phoneNumber') }}</span>
				</div>

				<div class="space-y-2">
					<label class="block text-sm font-medium text-foreground">{{ __('Amount (Ksh)') }}</label>
					<input type="number" step="0.01" min="{{ $this->depositLimits['min'] }}" @if($this->depositLimits['max'] < PHP_FLOAT_MAX) max="{{ $this->depositLimits['max'] }}" @endif wire:model="amount"
						class="w-full rounded-xl border border-border bg-background px-4 py-2.5 text-sm text-foreground"
						placeholder="0.00" />
					<p class="text-xs text-muted-foreground">
						{{ __('Min: :min', ['min' => number_format($this->depositLimits['min'], 2)]) }}
						@if($this->depositLimits['max'] < PHP_FLOAT_MAX)
							{{ __('Max: :max', ['max' => number_format($this->depositLimits['max'], 2)]) }}
						@endif
					</p>
					<span class="text-destructive text-xs">{{ $errors->first('amount') }}</span>
				</div>

				<button type="button" wire:click="submitDepositRequest" wire:loading.attr="disabled"
					class="w-full rounded-xl bg-[#2CB34A] px-4 py-2.5 text-sm font-semibold text-black disabled:cursor-not-allowed disabled:opacity-80">
					<span wire:loading.remove wire:target="submitDepositRequest">{{ __('Deposit with M-Pesa') }}</span>
					<span wire:loading wire:target="submitDepositRequest" class="inline-flex items-center gap-2">
						<i class="fa-solid fa-spinner animate-spin text-sm"></i>
						{{ __('Sending Request') }}
					</span>
				</button>

				<div class="flex items-center justify-center gap-5 text-xs text-muted-foreground">
					<span><i class="fa-solid fa-shield mr-1"></i>{{ __('Secure') }}</span>
					<span><i class="fa-solid fa-bolt mr-1"></i>{{ __('Instant') }}</span>
					<span><i class="fa-regular fa-circle-check mr-1"></i>{{ __('No Fees') }}</span>
				</div>
			</div>
		@else
			<div class="bg-card rounded-2xl border border-border p-5 sm:p-6 min-h-[52vh] flex flex-col">
				<div class="pt-3 space-y-4 text-center mb-4">
					<div class="mx-auto size-16 rounded-full bg-green-600 text-white flex items-center justify-center">
						<i class="fa-solid fa-circle-check text-2xl"></i>
					</div>
					<p class="text-lg font-bold text-foreground">{{ __('Payment Request Sent') }}</p>
				</div>

				<div class="my-auto space-y-6 rounded-xl border border-border bg-background p-5 mb-4">
					<p class="text-sm text-foreground">
						{{ __('Check your phone (:phone) and enter your M-Pesa PIN to complete the deposit of :amount.', ['phone' => $this->phoneNumber, 'amount' => number_format($submittedDepositAmount, 2)]) }}
					</p>
					<p class="text-sm font-semibold text-foreground">
						{{ __('This request expires in 60 seconds.') }}
					</p>
				</div>

				<div class="mt-auto grid grid-cols-2 gap-3">
					<button type="button" wire:click="resetDepositFlow"
						class="w-full rounded-xl bg-muted px-4 py-2.5 text-sm font-semibold text-muted-foreground">
						{{ __('Cancel') }}
					</button>
					<button type="button" wire:click="completePayment"
						class="w-full rounded-xl bg-[#2CB34A] px-4 py-2.5 text-sm font-semibold text-black">
						{{ __('I\'ve Completed Payment') }}
					</button>
				</div>
			</div>
		@endif
	</div>
</div>
