<?php

use App\Models\InvestmentPackage;
use App\Models\WalletTransaction;
use App\Jobs\Vrm\SendMail;
use App\Mail\AdminDepositMail;
use App\Mail\AdminTransactionMail;
use App\Mail\DepositRequestRejectedMail;
use App\Mail\WalletTransactionMail;
use App\Mail\WithdrawalRequestStatusMail;
use App\Services\Investment\SubscriptionService;
use App\Services\Sms\SmsNotifier;
use App\Services\Vrm\InvestmentEmailService;
use App\Traits\Vrm\Livewire\WithNotifications;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Schema;
use Livewire\Attributes\Computed;
use Livewire\Component;
use Livewire\WithPagination;

return new class extends Component {
    use WithNotifications;
    use WithPagination;

    public string $status = 'pending';

    #[Computed]
    public function depositRequests()
    {
        return WalletTransaction::query()
            ->with(['wallet', 'user'])
            ->whereIn('type', ['deposit_request', 'mpesa_deposit_request'])
            ->when($this->status !== 'all', fn ($query) => $query->where('status', $this->status))
            ->latest()
            ->paginate(12, ['*'], 'deposit_page');
    }

    #[Computed]
    public function withdrawalRequests()
    {
        return WalletTransaction::query()
            ->with(['wallet', 'user'])
            ->where('type', 'withdrawal_request')
            ->when($this->status !== 'all', fn ($query) => $query->where('status', $this->status))
            ->latest()
            ->paginate(12, ['*'], 'withdrawal_page');
    }

    public function approve(int $transactionId): void
    {
        $transaction = WalletTransaction::query()->with(['wallet', 'user'])->find($transactionId);

        if (! $transaction || $transaction->status !== 'pending') {
            $this->notifyError(__('Request not available for approval.'));

            return;
        }

        DB::transaction(function () use ($transaction): void {
            $wallet = $transaction->wallet;

            if (! $wallet) {
                throw new \RuntimeException('Wallet not found.');
            }

            if (in_array($transaction->type, ['deposit_request', 'mpesa_deposit_request'], true)) {
                $wallet->balance = (float) $wallet->balance + (float) $transaction->amount;
            }

            if ($transaction->type === 'withdrawal_request') {
                if (Schema::hasColumn('wallets', 'withdrawal_hold_amount')) {
                    $wallet->withdrawal_hold_amount = max(0, (float) $wallet->withdrawal_hold_amount - (float) $transaction->amount);
                } else {
                    $wallet->used_amount = max(0, (float) $wallet->used_amount - (float) $transaction->amount);
                }
                $wallet->balance = max(0, (float) $wallet->balance - (float) $transaction->amount);
            }

            $wallet->save();

            $transaction->status = 'approved';
            $transaction->approved_by = auth()->id();
            $transaction->approved_at = now();
            $transaction->save();
        });

        $notificationType = in_array($transaction->type, ['deposit_request', 'mpesa_deposit_request'], true) ? 'deposit' : 'withdrawal';
        if ($transaction->user?->email) {
            if ($notificationType === 'withdrawal') {
                SendMail::dispatch($transaction->user->email, mailable: new WithdrawalRequestStatusMail(
                    $transaction->user->name,
                    $transaction->currency_code,
                    (float) $transaction->amount,
                    'approved',
                    $transaction->transaction_code ?? data_get($transaction->meta, 'reference'),
                ));
            } else {
                SendMail::dispatch($transaction->user->email, mailable: new WalletTransactionMail(
                    $transaction->user->name,
                    $notificationType,
                    $transaction->currency_code,
                    (float) $transaction->amount,
                    'approved',
                    $transaction->phone_number ?: data_get($transaction->meta, 'phone_number', $transaction->user->phone),
                    $transaction->transaction_code ?? data_get($transaction->meta, 'reference'),
                    $transaction->updated_at?->format('M j, Y g:i A'),
                ));
            }
        }
        app(SmsNotifier::class)->sendToPhone(
            $transaction->phone_number ?: data_get($transaction->meta, 'phone_number', $transaction->user?->phone),
            ucfirst($notificationType).' request approved.',
        );

        $adminEmail = config('mail.admin_address');
        if (in_array($transaction->type, ['deposit_request', 'mpesa_deposit_request'], true) && filled($adminEmail)) {
            SendMail::dispatch((string) $adminEmail, mailable: new AdminDepositMail(
                    $transaction->user?->name ?? 'Unknown',
                    $transaction->user?->email ?? 'unknown@example.com',
                    (float) $transaction->amount,
                    $transaction->currency_code,
                    'approved',
                    $transaction->type === 'mpesa_deposit_request' ? 'mpesa' : 'wallet',
                    $transaction->transaction_code ?? data_get($transaction->meta, 'reference')
                ));

            app(SmsNotifier::class)->sendToAdmins(
                "Deposit request approved for ".($transaction->user?->name ?? 'member').": {$transaction->currency_code} ".number_format((float) $transaction->amount, 2).'.',
            );
        }

        if ($transaction->type === 'withdrawal_request' && filled($adminEmail)) {
            SendMail::dispatch((string) $adminEmail, mailable: new AdminTransactionMail(
                $transaction->user?->name ?? 'Unknown',
                $transaction->user?->email ?? 'unknown@example.com',
                (float) $transaction->amount,
                $transaction->currency_code,
                'withdrawal',
                'approved',
                (string) data_get($transaction->meta, 'channel', 'internal'),
                $transaction->transaction_code ?? data_get($transaction->meta, 'reference'),
            ));
        }

        if (in_array($transaction->type, ['deposit_request', 'mpesa_deposit_request'], true)) {
            $meta = $transaction->meta ?? [];
            if (($meta['intent'] ?? '') === SubscriptionService::META_INTENT_SUBSCRIPTION && ! empty($meta['package_id']) && isset($meta['amount'])) {
                $package = InvestmentPackage::query()->find($meta['package_id']);
                $user = $transaction->user;
                $wallet = $transaction->wallet;
                if ($package && $user && $wallet) {
                    $wallet->refresh();
                    try {
                        app(SubscriptionService::class)->createSubscription($user, $wallet, $package, (float) $meta['amount']);
                        $transaction->update(['meta' => array_merge($meta, ['subscription_fulfilled' => true])]);
                        app(InvestmentEmailService::class)->sendSubscriptionCreated(
                            $user->email,
                            $user->name,
                            $package->name,
                            app(\App\Services\DefaultCurrencyService::class)->code(),
                            (float) $meta['amount'],
                        );
                        app(SmsNotifier::class)->sendToPhone(
                            $user->phone,
                            "Subscription created for {$package->name}.",
                        );
                    } catch (\Throwable $e) {
                        Log::warning('Subscription after deposit approval failed', [
                            'transaction_id' => $transaction->id,
                            'message' => $e->getMessage(),
                        ]);
                    }
                }
            }
        }

        $this->notifySuccess(__('Wallet request approved.'));
    }

    public function reject(int $transactionId): void
    {
        $transaction = WalletTransaction::query()->with(['wallet', 'user'])->find($transactionId);

        if (! $transaction || $transaction->status !== 'pending') {
            $this->notifyError(__('Request not available for rejection.'));

            return;
        }

        DB::transaction(function () use ($transaction): void {
            $wallet = $transaction->wallet;

            if (! $wallet) {
                throw new \RuntimeException('Wallet not found.');
            }

            if ($transaction->type === 'withdrawal_request') {
                if (Schema::hasColumn('wallets', 'withdrawal_hold_amount')) {
                    $wallet->withdrawal_hold_amount = max(0, (float) $wallet->withdrawal_hold_amount - (float) $transaction->amount);
                } else {
                    $wallet->used_amount = max(0, (float) $wallet->used_amount - (float) $transaction->amount);
                }
                $wallet->save();
            }

            $transaction->status = 'rejected';
            $transaction->approved_by = auth()->id();
            $transaction->approved_at = now();
            $transaction->save();
        });

        $notificationType = in_array($transaction->type, ['deposit_request', 'mpesa_deposit_request'], true) ? 'deposit' : 'withdrawal';
        if ($transaction->user?->email) {
            if ($notificationType === 'deposit') {
                SendMail::dispatch($transaction->user->email, mailable: new DepositRequestRejectedMail(
                    $transaction->user->name,
                    $transaction->currency_code,
                    (float) $transaction->amount,
                    $transaction->transaction_code ?? data_get($transaction->meta, 'reference'),
                ));
            } elseif ($notificationType === 'withdrawal') {
                SendMail::dispatch($transaction->user->email, mailable: new WithdrawalRequestStatusMail(
                    $transaction->user->name,
                    $transaction->currency_code,
                    (float) $transaction->amount,
                    'rejected',
                    $transaction->transaction_code ?? data_get($transaction->meta, 'reference'),
                ));
            } else {
                SendMail::dispatch($transaction->user->email, mailable: new WalletTransactionMail(
                    $transaction->user->name,
                    $notificationType,
                    $transaction->currency_code,
                    (float) $transaction->amount,
                    'rejected',
                    $transaction->phone_number ?: data_get($transaction->meta, 'phone_number', $transaction->user->phone),
                    $transaction->transaction_code ?? data_get($transaction->meta, 'reference'),
                    $transaction->updated_at?->format('M j, Y g:i A'),
                ));
            }
        }
        app(SmsNotifier::class)->sendToPhone(
            $transaction->phone_number ?: data_get($transaction->meta, 'phone_number', $transaction->user?->phone),
            ucfirst($notificationType).' request rejected.',
        );

        $adminEmail = config('mail.admin_address');
        if (in_array($transaction->type, ['deposit_request', 'mpesa_deposit_request'], true) && filled($adminEmail)) {
            SendMail::dispatch((string) $adminEmail, mailable: new AdminDepositMail(
                $transaction->user?->name ?? 'Unknown',
                $transaction->user?->email ?? 'unknown@example.com',
                (float) $transaction->amount,
                $transaction->currency_code,
                'rejected',
                $transaction->type === 'mpesa_deposit_request' ? 'mpesa' : 'wallet',
                $transaction->transaction_code ?? data_get($transaction->meta, 'reference'),
            ));

            app(SmsNotifier::class)->sendToAdmins(
                "Deposit request rejected for ".($transaction->user?->name ?? 'member').": {$transaction->currency_code} ".number_format((float) $transaction->amount, 2).'.',
            );
        }

        if ($transaction->type === 'withdrawal_request' && filled($adminEmail)) {
            SendMail::dispatch((string) $adminEmail, mailable: new AdminTransactionMail(
                $transaction->user?->name ?? 'Unknown',
                $transaction->user?->email ?? 'unknown@example.com',
                (float) $transaction->amount,
                $transaction->currency_code,
                'withdrawal',
                'rejected',
                (string) data_get($transaction->meta, 'channel', 'internal'),
                $transaction->transaction_code ?? data_get($transaction->meta, 'reference'),
            ));
        }

        $this->notifySuccess(__('Wallet request rejected.'));
    }

    protected function view($data = [])
    {
        return app('view')->file('/home/ziidiaik/home.ziiditrader.com/storage/framework/views/livewire/views/0bcaf400.blade.php', $data);
    }
}; 