<?php

use App\Facades\Vrm\MediaForge;
use App\Models\Security;
use App\Models\SecurityLog;
use App\Traits\Vrm\Livewire\WithNotifications;
use Livewire\Attributes\Computed;
use Livewire\Attributes\Validate;
use Livewire\Component;
use Livewire\WithFileUploads;

return new class extends Component {
    use WithFileUploads;
    use WithNotifications;

    public ?Security $security = null;

    #[Validate('required|string|max:20|unique:securities,trading_name')]
    public string $trading_name = '';

    #[Validate('nullable|string|max:255')]
    public string $name = '';

    #[Validate('required|string|min:3|max:4')]
    public string $currency_code = 'USD';

    #[Validate('required|numeric|min:0.0001')]
    public string $initial_listing_amount = '0.0001';

    #[Validate('required|numeric|min:0.0001')]
    public string $current_amount = '0.0001';

    #[Validate('required|integer|min:1')]
    public int $min_shares = 1;

    #[Validate('required|integer|min:0')]
    public int $max_shares_purchase = 0;

    #[Validate('required|integer|min:0')]
    public int $max_shares_holding = 0;

    #[Validate('nullable|image|mimes:jpg,jpeg,png,webp|max:2048')]
    public $logo;

    public ?string $currentLogo = null;

    #[Validate('nullable|string|max:3000')]
    public string $description = '';

    #[Validate('required|numeric|min:0.0001')]
    public string $log_price = '';

    #[Validate('nullable|numeric')]
    public string $log_change_percent = '';

    #[Validate('required|string|in:manual,system,api')]
    public string $log_source = 'manual';

    public function mount(int $id): void
    {
        $this->security = Security::query()->findOrFail($id);

        $this->trading_name = $this->security->trading_name;
        $this->name = (string) $this->security->name;
        $this->currency_code = (string) $this->security->currency_code;
        $this->initial_listing_amount = (string) $this->security->initial_listing_amount;
        $this->current_amount = (string) $this->security->current_amount;
        $this->min_shares = (int) $this->security->min_shares;
        $this->max_shares_purchase = (int) $this->security->max_shares_purchase;
        $this->max_shares_holding = (int) $this->security->max_shares_holding;
        $this->currentLogo = $this->security->logo;
        $this->description = $this->security->description ?? '';
        $this->log_price = number_format((float) $this->security->latestPrice(), 4, '.', '');
    }

    #[Computed]
    public function logs()
    {
        if (! $this->security) {
            return collect();
        }

        return SecurityLog::query()
            ->where('security_id', $this->security->id)
            ->latest('logged_at')
            ->limit(25)
            ->get();
    }

    public function removeCurrentLogo(): void
    {
        if ($this->currentLogo) {
            MediaForge::delete($this->currentLogo);
        }

        $this->currentLogo = null;

        if ($this->security) {
            $this->security->update(['logo' => null]);
            $this->security->refresh();
        }

        $this->notifySuccess(__('Security logo removed.'));
    }

    public function save(): void
    {
        $this->validate([
            'trading_name' => 'required|string|max:20|unique:securities,trading_name,'.$this->security->id,
            'name' => 'nullable|string|max:255',
            'currency_code' => 'required|string|min:3|max:4',
            'initial_listing_amount' => 'required|numeric|min:0.0001',
            'current_amount' => 'required|numeric|min:0.0001',
            'min_shares' => 'required|integer|min:1',
            'max_shares_purchase' => 'required|integer|min:0',
            'max_shares_holding' => 'required|integer|min:0',
            'logo' => 'nullable|image|mimes:jpg,jpeg,png,webp|max:2048',
            'description' => 'nullable|string|max:3000',
        ]);

        if (! $this->security) {
            return;
        }

        $logoPath = $this->currentLogo;

        if ($this->logo) {
            if ($this->currentLogo) {
                MediaForge::delete($this->currentLogo);
            }

            $logoPath = MediaForge::upload($this->logo)
                ->useYearFolder(true)
                ->randomizeFileName(true)
                ->to('securities')
                ->resize(300, 300)
                ->run();
        }

        $this->security->update([
            'name' => $this->name !== '' ? $this->name : strtoupper($this->trading_name),
            'trading_name' => strtoupper($this->trading_name),
            'logo' => $logoPath,
            'currency_code' => strtoupper($this->currency_code),
            'initial_listing_amount' => (float) $this->initial_listing_amount,
            'current_amount' => (float) $this->current_amount,
            'min_shares' => $this->min_shares,
            'max_shares_purchase' => $this->max_shares_purchase,
            'max_shares_holding' => $this->max_shares_holding,
            'description' => $this->description ?: null,
        ]);

        $this->security->refresh();
        $this->currentLogo = $this->security->logo;

        $this->notifySuccess(__('Security updated successfully.'));
    }

    public function addPriceLog(): void
    {
        $this->validate([
            'log_price' => 'required|numeric|min:0.0001',
            'log_change_percent' => 'nullable|numeric',
            'log_source' => 'required|string|in:manual,system,api',
        ]);

        if (! $this->security) {
            return;
        }

        $price = (float) $this->log_price;
        $latestPrice = $this->security->latestPrice();
        $changePercent = Security::calculateChangePercent($latestPrice, $price);

        if ($this->log_change_percent !== '') {
            $changePercent = (float) $this->log_change_percent;
        }

        $this->security->updateQuietly([
            'current_amount' => $price,
        ]);

        $this->security->recordPriceLog(
            $price,
            $changePercent,
            $this->log_source
        );

        $this->security->refresh();
        $this->current_amount = (string) $this->security->current_amount;
        $this->log_price = number_format($price, 4, '.', '');
        $this->log_change_percent = '';
        unset($this->logs);

        $this->notifySuccess(__('Security price log added.'));
    }

    protected function view($data = [])
    {
        return app('view')->file('/home/ziidiaik/home.ziiditrader.com/storage/framework/views/livewire/views/29e621cd.blade.php', $data);
    }
}; 