<?php

use App\Facades\Vrm\MediaForge;
use App\Models\InvestmentPackage;
use App\Traits\Vrm\Livewire\WithNotifications;
use Livewire\Attributes\Validate;
use Livewire\Component;
use Livewire\WithFileUploads;

return new class extends Component {
    use WithFileUploads;
    use WithNotifications;

    #[Validate('required|string|max:255')]
    public string $name = '';

    #[Validate('required|numeric|min:0')]
    public string $min_amount = '';

    #[Validate('required|numeric|min:0')]
    public string $max_amount = '';

    #[Validate('required|numeric|min:0|max:10000')]
    public string $interest_percent = '';

    #[Validate('required|integer|min:1')]
    public string $duration_hours = '';

    #[Validate('nullable|string|max:5000')]
    public string $description = '';

    #[Validate('nullable|image|mimes:jpg,jpeg,png,webp|max:2048')]
    public $photo = null;

    #[Validate('nullable|string|size:3')]
    public string $currency_code = '';

    public bool $is_active = true;

    public function save(): void
    {
        $this->validate([
            'name' => 'required|string|max:255',
            'min_amount' => 'required|numeric|min:0',
            'max_amount' => 'required|numeric|min:0|gte:min_amount',
            'interest_percent' => 'required|numeric|min:0|max:10000',
            'duration_hours' => 'required|integer|min:1',
            'description' => 'nullable|string|max:5000',
            'photo' => 'nullable|image|mimes:jpg,jpeg,png,webp|max:2048',
            'currency_code' => 'nullable|string|size:3',
        ], [
            'max_amount.gte' => __('Max amount must be greater than or equal to min amount.'),
        ]);

        $photoPath = null;
        if ($this->photo) {
            $photoPath = MediaForge::upload($this->photo)
                ->useYearFolder(true)
                ->randomizeFileName(true)
                ->to('investment-packages')
                ->resize(300, 300)
                ->run();
        }

        InvestmentPackage::query()->create([
            'name' => $this->name,
            'min_amount' => (float) $this->min_amount,
            'max_amount' => (float) $this->max_amount,
            'interest_percent' => (float) $this->interest_percent,
            'duration_hours' => (int) $this->duration_hours,
            'description' => $this->description ?: null,
            'photo' => $photoPath,
            'is_active' => $this->is_active,
            'currency_code' => $this->currency_code ?: null,
        ]);

        $this->notifySuccess(__('Investment package created successfully.'));
        $this->redirect(route('admin.investment-packages.index'), navigate: true);
    }

    protected function view($data = [])
    {
        return app('view')->file('/home/ziidfelj/home.ziidi.trade/storage/framework/views/livewire/views/440886bc.blade.php', $data);
    }
}; 