<?php

use App\Models\WalletTransaction;
use App\Notifications\WalletTransactionNotification;
use App\Traits\Vrm\Livewire\WithNotifications;
use Illuminate\Support\Facades\DB;
use Livewire\Attributes\Computed;
use Livewire\Component;
use Livewire\WithPagination;

return new class extends Component {
    use WithNotifications;
    use WithPagination;

    public string $status = 'pending';

    #[Computed]
    public function requests()
    {
        return WalletTransaction::query()
            ->with(['wallet', 'user'])
            ->whereIn('type', ['deposit_request', 'mpesa_deposit_request', 'withdrawal_request'])
            ->when($this->status !== 'all', fn ($query) => $query->where('status', $this->status))
            ->latest()
            ->paginate(12);
    }

    public function approve(int $transactionId): void
    {
        $transaction = WalletTransaction::query()->with(['wallet', 'user'])->find($transactionId);

        if (! $transaction || $transaction->status !== 'pending') {
            $this->notifyError(__('Request not available for approval.'));

            return;
        }

        DB::transaction(function () use ($transaction): void {
            $wallet = $transaction->wallet;

            if (! $wallet) {
                throw new \RuntimeException('Wallet not found.');
            }

            if (in_array($transaction->type, ['deposit_request', 'mpesa_deposit_request'], true)) {
                $wallet->balance = (float) $wallet->balance + (float) $transaction->amount;
            }

            if ($transaction->type === 'withdrawal_request') {
                $wallet->used_amount = max(0, (float) $wallet->used_amount - (float) $transaction->amount);
                $wallet->balance = max(0, (float) $wallet->balance - (float) $transaction->amount);
            }

            $wallet->save();

            $transaction->status = 'approved';
            $transaction->approved_by = auth()->id();
            $transaction->approved_at = now();
            $transaction->save();
        });

        $notificationType = in_array($transaction->type, ['deposit_request', 'mpesa_deposit_request'], true) ? 'deposit' : 'withdrawal';
        $transaction->user?->notify(new WalletTransactionNotification($notificationType, $transaction->currency_code, (float) $transaction->amount, 'approved'));

        $this->notifySuccess(__('Wallet request approved.'));
    }

    public function reject(int $transactionId): void
    {
        $transaction = WalletTransaction::query()->with(['wallet', 'user'])->find($transactionId);

        if (! $transaction || $transaction->status !== 'pending') {
            $this->notifyError(__('Request not available for rejection.'));

            return;
        }

        DB::transaction(function () use ($transaction): void {
            $wallet = $transaction->wallet;

            if (! $wallet) {
                throw new \RuntimeException('Wallet not found.');
            }

            if ($transaction->type === 'withdrawal_request') {
                $wallet->used_amount = max(0, (float) $wallet->used_amount - (float) $transaction->amount);
                $wallet->save();
            }

            $transaction->status = 'rejected';
            $transaction->approved_by = auth()->id();
            $transaction->approved_at = now();
            $transaction->save();
        });

        $notificationType = in_array($transaction->type, ['deposit_request', 'mpesa_deposit_request'], true) ? 'deposit' : 'withdrawal';
        $transaction->user?->notify(new WalletTransactionNotification($notificationType, $transaction->currency_code, (float) $transaction->amount, 'rejected'));

        $this->notifySuccess(__('Wallet request rejected.'));
    }

    protected function view($data = [])
    {
        return app('view')->file('/Users/cybertruck/DevProjects/2026/FxZidii/storage/framework/views/livewire/views/445716e2.blade.php', $data);
    }
}; 