<?php

use App\Models\NotificationState;
use Illuminate\Contracts\Pagination\LengthAwarePaginator;
use Illuminate\Database\Eloquent\Builder;
use Livewire\Attributes\Computed;
use Livewire\Attributes\Layout;
use Livewire\Component;
use Livewire\WithPagination;

return new #[Layout('layouts.trading')] class extends Component {
    use WithPagination;

    public string $status = 'all';

    public string $search = '';

    /**
     * @var array<int, int>
     */
    public array $selectedStateIds = [];

    public function mount(): void
    {
        NotificationState::query()
            ->where('user_id', auth()->id())
            ->whereNull('seen_at')
            ->whereNull('deleted_at')
            ->update(['seen_at' => now()]);
    }

    public function updatedStatus(): void
    {
        $this->resetPage('notifications_page');
        $this->selectedStateIds = [];
    }

    public function updatedSearch(): void
    {
        $this->resetPage('notifications_page');
        $this->selectedStateIds = [];
    }

    #[Computed]
    public function unreadCount(): int
    {
        return $this->baseStatesQuery()
            ->whereNull('read_at')
            ->count();
    }

    #[Computed]
    public function readCount(): int
    {
        return $this->baseStatesQuery()
            ->whereNotNull('read_at')
            ->count();
    }

    #[Computed]
    public function notifications(): LengthAwarePaginator
    {
        $query = $this->baseStatesQuery()
            ->with('notification')
            ->whereHas('notification', function (Builder $query): void {
                $query->where('is_active', true);
            });

        if ($this->status === 'unread') {
            $query->whereNull('read_at');
        }

        if ($this->status === 'read') {
            $query->whereNotNull('read_at');
        }

        if ($this->search !== '') {
            $searchValue = '%'.$this->search.'%';

            $query->whereHas('notification', function (Builder $query) use ($searchValue): void {
                $query
                    ->where('title', 'like', $searchValue)
                    ->orWhere('body', 'like', $searchValue)
                    ->orWhere('type', 'like', $searchValue);
            });
        }

        return $query->latest('id')->paginate(15, ['*'], 'notifications_page');
    }

    public function markAsRead(int $stateId): void
    {
        NotificationState::query()
            ->where('id', $stateId)
            ->where('user_id', auth()->id())
            ->whereNull('deleted_at')
            ->whereNull('read_at')
            ->update(['read_at' => now()]);

        $selectedIndex = array_search($stateId, $this->selectedStateIds, true);

        if ($selectedIndex !== false) {
            unset($this->selectedStateIds[$selectedIndex]);
            $this->selectedStateIds = array_values($this->selectedStateIds);
        }
    }

    public function markSelectedAsRead(): void
    {
        if ($this->selectedStateIds === []) {
            return;
        }

        NotificationState::query()
            ->where('user_id', auth()->id())
            ->whereNull('deleted_at')
            ->whereNull('read_at')
            ->whereIn('id', $this->selectedStateIds)
            ->update(['read_at' => now()]);

        $this->selectedStateIds = [];
    }

    public function markAllAsRead(): void
    {
        $this->baseStatesQuery()
            ->whereNull('read_at')
            ->update(['read_at' => now()]);

        $this->selectedStateIds = [];
    }

    private function baseStatesQuery(): Builder
    {
        return NotificationState::query()
            ->where('user_id', auth()->id())
            ->where('is_active', true)
            ->whereNull('cleared_at')
            ->whereNull('deleted_at');
    }

    protected function view($data = [])
    {
        return app('view')->file('/home/ziidfelj/home.ziidi.trade/storage/framework/views/livewire/views/94fd64b1.blade.php', $data);
    }
}; 