<?php

use App\Models\User;
use App\Models\Vrm\Role;
use App\Notifications\PasswordUpdatedNotification;
use App\Notifications\VerifyEmail;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Notification;
use Livewire\Livewire;

function ensureMemberRoleForAccountSettings(): Role
{
    return Role::query()->firstOrCreate(
        ['id' => 3],
        [
            'name' => 'Member',
            'slug' => 'member',
            'module' => 'contribution',
            'authority' => 'member',
            'is_active' => true,
            'description' => 'Member',
        ]
    );
}

test('member can update profile information from account page', function () {
    Notification::fake();

    $memberRole = ensureMemberRoleForAccountSettings();

    $member = User::factory()->create([
        'email_verified_at' => now(),
    ]);
    $member->roles()->attach($memberRole->id);

    $this->actingAs($member);

    Livewire::test('front.account')
        ->set('name', 'Updated Member')
        ->set('email', 'updated.member@example.com')
        ->call('updateProfileInformation');

    $member->refresh();

    expect($member->name)->toBe('Updated Member');
    expect($member->email)->toBe('updated.member@example.com');
    expect($member->email_verified_at)->toBeNull();

    Notification::assertSentTo($member, VerifyEmail::class);
});

test('member can update password from account page', function () {
    Notification::fake();

    $memberRole = ensureMemberRoleForAccountSettings();

    $member = User::factory()->create();
    $member->roles()->attach($memberRole->id);

    $this->actingAs($member);

    Livewire::test('front.account')
        ->set('current_password', 'password')
        ->set('password', 'new-password-123')
        ->set('password_confirmation', 'new-password-123')
        ->call('updatePassword')
        ->assertHasNoErrors();

    $member->refresh();

    expect(Hash::check('new-password-123', $member->password))->toBeTrue();

    Notification::assertSentTo($member, PasswordUpdatedNotification::class);
});
